% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ifun.R
\name{ifun}
\alias{ifun}
\title{Invert an expression defining a data transformation}
\usage{
ifun(expr, verbose = FALSE)
}
\arguments{
\item{expr}{a single-variable call or quoted expression to be inverted.
The variable's name in \code{expr} is referred to here as \code{varname}.}

\item{verbose}{a logical controlling printing of the intermediate functions
\eqn{f(.)}, \eqn{g(.)}, \eqn{h(.)} etc (see 'Details').}
}
\value{
The required inverting function, with single argument \code{x}. Its
\code{"varname"} attribute contains \code{varname} as a character string.
}
\description{
Given a transformed variable and the expression used to transform it, \code{ifun} creates
a function containing the inverse expression that will back-transform the variable.
}
\details{
\code{ifun} returns the inverting function such that
\code{ifun(expr)(eval(expr)) = varname}, where
\code{expr} can include any of the invertible functions in the \code{Math} and \code{Ops}
groups, plus \code{identity} and \code{I}.

To illustrate its use, consider variants of the \code{sitar} model
\code{height ~ age} where \code{age} and/or \code{height} are transformed,
e.g. \code{height ~ log(age)} or \code{log(height) ~ sqrt(age)}. Each model
is of the form \code{y ~ x} but the units of \code{x} and \code{y} vary.

The models are compared by plotting the fitted curves in their original units,
by first applying suitable functions to back-transform \code{x} and \code{y}.
For example with \code{log(age)}, where \code{expr = quote(log(age))},
the function \code{ifun = function(x) exp(x)} back-transforms
\code{eval(expr)} to give \code{age}. See the first example.

\code{ifun} generalises this process for increasingly complex \code{expr}, as
the next two examples show.

The final example shows \code{ifun} in action with \code{\link{plot.sitar}},
which uses \code{ifun} as the default function for arguments \code{xfun} and
\code{yfun} - they are used to back-transform \code{x} and \code{y} using the
values of \code{expr} for \code{x} and \code{y} extracted from the model's
\code{sitar} call.

Structuring \code{expr} suitably ensures it can be inverted - it should
contain a single mention of a single variable (\code{varname} here),
and possibly functions such as \eqn{f(.)}, \eqn{g(.)}, \eqn{h(.)} etc
such that \code{expr} = \eqn{f(g(h((varname))))}. The number of such functions
is in principle unlimited. \code{ifun} returns \code{function(x)}
\eqn{h^{-1}(g^{-1}(f^{-1}((x))))}{h^-1(g^-1(f^-1((x))))},
which ensures that
\code{expr} is invertible so long as the individual functions are invertible.
}
\examples{
## for best effect run all the code

## define varname variable
(age <- 1:9)

## simple case - age transformed to log(age)
(expr <- quote(log(age)))
## transformed age
eval(expr)
## inverting function, with "varname" attribute set to "age"
ifun(expr)
## inverted transformed age identical to age
all.equal(age, ifun(expr)(eval(expr)))

## more complex case - age transformed to log age since conception
(expr <- quote(log(age + 0.75)))
## inverting function
ifun(expr)
## inverted transformed age identical to age
all.equal(age, ifun(expr)(eval(expr)))

## ludicrously complex case involving exp, log10, ^, pi and trigonometry
(expr <- quote((exp(sin(pi * log10(age + 0.75)/2) - 1)^4)))
## inverting function, showing intermediate stages
ifun(expr, verbose=TRUE)
## identical to original
all.equal(age, ifun(expr)(eval(expr)))

## example of plot.sitar back-transforming transformed x and y in sitar models
## fit sitar models
m1 <- sitar(x=age, y=height^2, id=id, data=heights, df=6)
m2 <- update(m1, x=log(age+0.75), y=height)

## default plot options for xfun & yfun back-transform x & y to original scales
## xfun=ifun(x$call.sitar$x)
## yfun=ifun(x$call.sitar$y)
## compare mean curves for the two models where x & y are on the original scales
plot(m1, 'd', las=1)
lines(m2, 'd', col=2)
}
\seealso{
\code{\link{plot.sitar}}
}
\author{
Tim Cole \email{tim.cole@ucl.ac.uk}
}
