% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_risk.R
\name{riskratio}
\alias{riskratio}
\alias{riskdiff}
\title{Fit risk ratio and risk difference models}
\usage{
riskratio(
  formula,
  data,
  approach = c("auto", "all", "robpoisson", "duplicate", "glm", "glm_startp",
    "glm_startd", "glm_cem", "glm_cem_startp", "margstd_boot", "margstd_delta",
    "logistic", "legacy"),
  variable = NULL,
  at = NULL,
  ...
)

riskdiff(
  formula,
  data,
  approach = c("auto", "all", "robpoisson", "glm", "glm_startp", "glm_cem",
    "glm_cem_startp", "margstd_boot", "margstd_delta", "legacy"),
  variable = NULL,
  at = NULL,
  ...
)
}
\arguments{
\item{formula}{A formula object of the form \code{response ~ predictors}.}

\item{data}{A \code{tibble} or \code{data.frame} object.}

\item{approach}{Optional: Method for model fitting.
\itemize{
\item \code{"auto"} (default) is recommended; it will return results of
\code{"margstd_delta"} unless interaction terms between exposure and
confounders are included. This these cases, results from
\code{"margstd_boot"} are returned.
\item \code{"all"} will attempt to fit
the model via all implemented approaches to allow for comparisons.
\item \code{"legacy"} selects the most efficient approach that converges and
ensures that predicted probabilities are within range (< 1).
}

The other options allow for directly selecting a fitting approach,
some of which may not converge or yield out-of-range predicted
probabilities. See full documentation for details.
\itemize{
\item \code{"glm"} Binomial model.
\item \code{"glm_startp"} Binomial model with starting values from Poisson
model.
\item \code{"glm_startd"} Binomial model with starting values from logistic
model with case duplication.
\item \code{"robpoisson"} Poisson model with robust covariance.
\item \code{"duplicate"} Logistic model with duplication of cases. Only
available in \code{riskratio()}.
\item \code{"glm_cem"} Binomial model fitted with combinatorial expectation
maximization.
\item \code{"glm_cem_startp"} As \code{glm_cem}, with Poisson starting values.
\item \code{"margstd_boot"} Marginal standardization after logistic model,
bootstrap standard errors/confidence intervals.
\item \code{"margstd_delta"} Marginal standardization after logistic model,
delta method standard errors/confidence intervals.
\item \code{"logistic"} For comparison only: the logistic model. Only available
in \code{riskratio()}.
}}

\item{variable}{Optional: exposure variable to use for marginal
standardization. If \code{variable} is not provided and marginal
standardization is attempted, then the first variable in the model
is used as the exposure. Levels are determined automatically for
variables types \code{logical}, \code{character}, \code{factor} and can
optionally be supplied via \code{at =}.}

\item{at}{Optional: Levels of exposure variable \code{variable} for marginal
standardization. \code{at =} determines the levels at which contrasts of
the exposure are to be assessed. The level listed first is used as the
reference. Levels must exist in the data for character, factor or ordered
factor variables. For numeric variables, levels that do not exist in the
data can be interpolations or extrapolations; if levels exceed the
extremes of the data (extrapolation), a warning will be displayed.}

\item{...}{Optional: Further arguments passed to fitting functions
(\code{glm}, \code{logbin}, or \code{addreg}).}
}
\value{
Fitted model. This object can be passed on to post-processing
functions:
\itemize{
\item \code{\link[risks]{summary.risks}}: an overview of results
(risks-specific S3 methods: \code{\link[risks]{summary.robpoisson}},
\code{\link[risks]{summary.margstd_boot}},
\code{\link[risks]{summary.margstd_delta}}).
\item \code{\link[risks]{tidy.risks}}: a tibble of coefficients and confidence
intervals.
}

Standard post-processing functions can also be used:
\itemize{
\item \code{\link[stats]{coef}}: a vector of coefficients.
\item \code{\link[stats]{confint}}: a matrix of confidence intervals
(risks-specific S3 methods: \code{\link[risks]{confint.robpoisson}},
\code{\link[risks]{confint.margstd_boot}},
\code{\link[risks]{confint.margstd_delta}}).
\item \code{\link[stats]{predict.glm}(type = "response")}: fitted values
(predictions).
\item \code{\link[stats]{residuals}}: residuals.
}

If model fitting using all possible approaches was requested via
\code{approach = "all"}, then their results can be retrieved from the
list \code{all_models} in the returned object (e.g.,
\code{fit$all_models[[1]]}, \code{fit$all_models[[2]]}, etc.).
}
\description{
\code{riskratio} and \code{riskdiff} provide a flexible interface to fitting
risk ratio and risk difference models.

In cohort studies with a binary outcome, risk ratios and risk differences
are typically more appropriate to report than odds ratios from logistic
regression, yet such models have historically been difficult to implement in
standard software.

The risks package selects an efficient way to fit risk ratio or
risk difference models successfully, which will converge whenever logistic
models converge. Optionally, a specific approach to model fitting can also be
requested. Implemented are Poisson models with robust covariance, binomial
models, logistic models with case duplication, binomial models aided in
convergence by starting values obtained through Poisson models or
logistic models with case duplication, binomial models fitted via
combinatorial expectation maximization (optionally also with Poisson starting
values), and estimates obtained via marginal standardization after logistic
regression with bootstrapped or delta method for confidence intervals.

Adjusting for covariates (e.g., confounders) in the model specification
(\code{formula =}) is possible.
}
\section{Functions}{
\itemize{
\item \code{riskratio()}: Fit risk ratio models

\item \code{riskdiff()}: Fit risk difference models

}}
\examples{
data(breastcancer)  # Cohort study with binary outcome
                    # See for details: help(breastcancer)

# Risk ratio model
fit_rr <- riskratio(formula = death ~ stage + receptor, data = breastcancer)
fit_rr
summary(fit_rr)

# Risk difference model
fit_rd <- riskdiff(formula = death ~ stage + receptor, data = breastcancer)
fit_rd
summary(fit_rd)
}
\references{
Wacholder S. Binomial regression in GLIM: Estimating risk ratios
and risk differences. Am J Epidemiol 1986;123:174-184.
(Binomial regression models; \code{approach = "glm"})

Spiegelman D, Hertzmark E. Easy SAS Calculations for Risk or
Prevalence Ratios and Differences. Am J Epidemiol 2005;162:199-200.
(Binomial models fitted used starting values from Poisson models;
\code{approach = "glm_start"})

Zou G. A modified Poisson regression approach to prospective
studies with binary data. Am J Epidemiol 2004;159:702-706.
(Poisson model with robust/sandwich standard errors;
\code{approach = "robpoisson"})

Schouten EG, Dekker JM, Kok FJ, Le Cessie S, Van Houwelingen HC,
Pool J, Vandenbroucke JP. Risk ratio and rate ratio estimation in
case-cohort designs: hypertension and cardiovascular mortality.
Stat Med 1993;12:1733–45; (Logistic model with case duplication and
cluster-robust standard errors, \code{approach = "duplicate"}).

Donoghoe MW, Marschner IC. logbin: An R Package for
Relative Risk Regression Using the Log-Binomial Model.
J Stat Softw 2018;86(9). (Log-binomial models fitted via combinatorial
expectation maximization; \code{riskratio(approach = "glm_cem")}

Donoghoe MW, Marschner IC. Stable computational methods
for additive binomial models with application to adjusted risk differences.
Comput Stat Data Anal 2014;80:184-96. (Additive binomial models
fitted via combinatorial expectation maximization;
\code{riskdiff(approach = "glm_cem")})

Localio AR, Margolis DJ, Berlin JA.
Relative risks and confidence intervals were easily computed
indirectly from multivariable logistic regression.
J Clin Epidemiol 2007;60(9):874-82. (Marginal standardization after fitting
a logistic model; \code{approach = "margstd_boot"})
}
