#' Load plain-text data from Dropbox non-Public folders
#' 
#' \code{source_DropboxData} loads plain-text formatted data stored on Dropbox in a non-Public folder.
#' @param file The plain-text formatted data's file name as a character string.
#' @param key The file's Dropbox key as a character string. 
#' @param sha1 Character string of the file's SHA-1 hash, generated by \code{source_data}.
#' @param sep The separator method for the data. For example, to load comma-separated values data (CSV) use \code{sep = ","} (the default). To load tab-separated values data (TSV) use \code{sep = "\t"}.
#' @param header Logical, whether or not the first line of the file is the header (i.e. variable names). The default is \code{header = TRUE}.
#' @return a data frame
#' @details Loads plain-text data (e.g. CSV, TSV) data from a Dropbox non-public folder. To download data from a Drobpox Public folder simply use \code{read.table}, giving the public URL as the file name.
#' 
#' To download data using \code{source_DropboxData} you will need (a) the file's name and (b) its Dropbox key. The key can be found by clicking "Share Link". The key will then be listed as part of the URL directly after "https://www.dropbox.com/s/" and before the file name.
#'
#' From the source_url documentation: "If a SHA-1 hash is specified with the sha1 argument, then this function will check the SHA-1 hash of the downloaded file to make sure it matches the expected value, and throw an error if it does not match. If the SHA-1 hash is not specified, it will print a message displaying the hash of the downloaded file. The purpose of this is to improve security when running remotely-hosted code; if you have a hash of the file, you can be sure that it has not changed."
#' @examples
#' # Download Financial Regulatory Governance data (see Gandrud 2012)
#' FinData <- source_DropboxData(file = "fin_research_note.csv",
#'								 key = "exh4iobbm2p5p1v",
#'				 				 sep = ",", header = TRUE)
#'
#' @source Based on \code{source_url} from the Hadley Wickham's devtools package and \code{url_dl} from the qdap package.
#' Data from: Gandrud, Christopher. 2012. "The Diffusion of Financial Supervisory Governance Ideas." Review of International Political Economy: 1-36.
#' @seealso \link{httr} and \code{\link{read.table}}
#' @import digest
#' @importFrom httr GET stop_for_status text_content content
#' @export

source_DropboxData <-function(file, key, sha1 = NULL, sep = ",", header = TRUE)
{
	URL <- paste0('https://dl.dropboxusercontent.com/s/', 
					key, '/', file)
	
	stopifnot(is.character(URL), length(URL) == 1)

    temp_file <- tempfile()
    on.exit(unlink(temp_file))
    
    request <- GET(URL)
    stop_for_status(request)
    writeBin(content(request, type = "raw"), temp_file)
    
    file_sha1 <- digest(file = temp_file, algo = "sha1")
    
    if (is.null(sha1)) {
        message("SHA-1 hash of file is ", file_sha1)
    }
    else {
        if (!identical(file_sha1, sha1)) {
            stop("SHA-1 hash of downloaded file (", file_sha1, 
                ")\n  does not match expected value (", sha1, 
                ")", call. = FALSE)
        }
    }
	
	read.table(temp_file, sep = sep, header = header)
}