% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vital_rate_driver_simulation.R
\name{driven_vital_rate}
\alias{driven_vital_rate}
\title{Calculate driven vital rates}
\usage{
driven_vital_rate(
  driver,
  baseline_value = NULL,
  slope = NULL,
  baseline_driver = NULL,
  error_sd = 0,
  link = "logit"
)
}
\arguments{
\item{driver}{A vector of driver values.}

\item{baseline_value}{A vector or matrix of baseline values for the vital
rate (e.g., survival) that is being influenced ("driven") by another
variable (e.g. a climatic variable).}

\item{slope}{A vector or matrix of slopes for the relationship between the
driver variable and the vital rate being driven.}

\item{baseline_driver}{The \code{baseline_driver} parameter is a single value
representing the baseline driver value. If the driver value is greater than
this value and the slope is positive, then the resulting vital rate will be
higher. Conversely, if the driver value is less than this variable and the
slope is positive, then the resulting vital rate will be less than the
baseline value.}

\item{error_sd}{A vector or matrix of error standard deviations for random
normal error to be added to the driven value of the vital rate being
modelled. If set to 0 (the default), no error is added.}

\item{link}{A character string indicating the type of link function to use.
Valid values are "\code{logit}" (the default) and "\code{log}", which are appropriate
for survival (U submatrix) and fecundity (F submatrix) respectively.}
}
\value{
Depending on the input types, either a single value, a vector or a
list of matrices of driven values for the vital rate(s) being modelled. The
list has a length equal to the length of the \code{driver} input parameter.
}
\description{
This function calculates new values for a vital rate, such as survival or
fecundity that is being influenced by a driver (e.g., weather). It does this
by using a driver variable and a baseline value, along with a specified slope
for the relationship between the driver variable and the vital rate. The
function works on a linearised scale, using logit for survival and log for
fecundity, and takes into account the error standard deviation.
}
\details{
The relationship between the driver variable and the vital rate is assumed to
be linear:

$$V = a * (d - d_b) + x + E$$

Where $$V$$ is the new vital rate (on the scale of the linear predictor),
$$a$$ is the slope, $$x$$ is the baseline vital rate, $$d$$ is the driver,
$$d_b$$ is the baseline driver and $$E$$ is the error.

The input vital rate(s) (\code{baseline_value}) can be a single-element vector
representing a single vital rate (e.g., survival probability or fecundity), a
longer vector representing a series of vital rates (e.g., several survival
probabilities or fecundity values), or a matrix of values (e.g., a U or F
submatrix of a matrix population model). The \code{slope}s of the relationship
between the vital rate (\code{baseline_value}) and the driver can be provided as a
single value, which is applied to all elements of the input vital rates, or
as a matrix of values that map onto the matrix of vital rates. This allows
users to simulate cases where different vital rates in a matrix model are
affected in different ways by the same weather driver. For example, juvenile
survival might be more affected by the driver than adult survival. The
\code{baseline_driver} value represents the "normal" state of the driver. If the
driver is greater than the \code{baseline_driver} and the \code{slope} is positive,
then the outcome vital rate will be higher. If the driver is less than the
\code{baseline_driver} variable and the \code{slope} is positive, then the outcome
vital rate will be less than the \code{baseline_value.} The \code{error_sd} represents
the error in the linear relationship between the driver and the vital rate.
}
\examples{
set.seed(42) # set seed for repeatability

# A single vital rate and a single driver
driven_vital_rate(
  driver = 14,
  baseline_value = 0.5,
  slope = .4,
  baseline_driver = 10,
  error_sd = 0,
  link = "logit"
)

# A single vital rate and a time series of drivers
driven_vital_rate(
  driver = runif(10, 5, 15),
  baseline_value = 0.5,
  slope = .4,
  baseline_driver = 10,
  error_sd = 0,
  link = "logit"
)

# A matrix of survival values (U submatrix of a Leslie model)
# with a series of drivers, and matrices of slopes and errors

lt1 <- model_survival(params = c(b_0 = 0.4, b_1 = 0.5), model = "Gompertz")
lt1$fecundity <- model_fecundity(
  age = 0:max(lt1$x), params = c(A = 10),
  maturity = 3, model = "step"
)

mats <- make_leslie_mpm(
  survival = lt1$px, fecundity = lt1$fecundity, n_stages =
    nrow(lt1), split = TRUE
)
mats$mat_U
mat_dim <- nrow(mats$mat_U)

driven_vital_rate(
  driver = runif(5, 5, 15),
  baseline_value = mats$mat_U,
  slope = matrix(.4,
    nrow = mat_dim,
    ncol = mat_dim
  ),
  baseline_driver = 10,
  error_sd = matrix(1, nrow = mat_dim, ncol = mat_dim),
  link = "logit"
)

}
\author{
Owen Jones \href{mailto:jones@biology.sdu.dk}{jones@biology.sdu.dk}
}
\concept{drivers}
