% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/var_label.R
\name{var_label}
\alias{var_label}
\alias{var_label.data.frame}
\alias{var_label<-}
\alias{set_variable_labels}
\title{Get / Set a variable label}
\usage{
var_label(x, ...)

\method{var_label}{data.frame}(x, unlist = FALSE, null_action = c("keep", "fill", "skip"), ...)

var_label(x) <- value

set_variable_labels(.data, ..., .labels = NA, .strict = TRUE)
}
\arguments{
\item{x}{a vector or a data.frame}

\item{...}{name-value pairs of variable labels (see examples)}

\item{unlist}{for data frames, return a named vector instead of a list}

\item{null_action}{for data frames, by default \code{NULL} will be returned for
columns with no variable label. Use \code{"fill"} to populate with the column name
instead, or \code{"skip"} to remove such values from the returned list.}

\item{value}{a character string or \code{NULL} to remove the label
For data frames, it could also be a named list or a character vector
of same length as the number of columns in \code{x}.}

\item{.data}{a data frame or a vector}

\item{.labels}{variable labels to be applied to the data.frame,
using the same syntax as \code{value} in \code{var_label(df) <- value}.}

\item{.strict}{should an error be returned if some labels
doesn't correspond to a column of \code{x}?}
}
\value{
\code{set_variable_labels()} will return an updated copy of \code{.data}.
}
\description{
Get / Set a variable label
}
\details{
For data frames, if \code{value} is a named list, only elements whose name will
match a column of the data frame will be taken into account. If \code{value}
is a character vector, labels should in the same order as the columns of
the data.frame.
}
\note{
\code{set_variable_labels()} could be used with \pkg{dplyr} syntax.
}
\examples{
var_label(iris$Sepal.Length)
var_label(iris$Sepal.Length) <- 'Length of the sepal'
\dontrun{
 View(iris)
}
# To remove a variable label
var_label(iris$Sepal.Length) <- NULL
# To change several variable labels at once
var_label(iris) <- c(
  "sepal length", "sepal width", "petal length",
  "petal width", "species"
)
var_label(iris)
var_label(iris) <- list(
  Petal.Width = "width of the petal",
  Petal.Length = "length of the petal",
  Sepal.Width = NULL,
  Sepal.Length = NULL
)
var_label(iris)
var_label(iris, null_action = "fill")
var_label(iris, null_action = "skip")
var_label(iris, unlist = TRUE)

if (require(dplyr)) {
  # adding some variable labels
  df <- tibble(s1 = c("M", "M", "F"), s2 = c(1, 1, 2)) \%>\%
    set_variable_labels(s1 = "Sex", s2 = "Yes or No?")
  var_label(df)

  # removing a variable label
  df <- df \%>\% set_variable_labels(s2 = NULL)
  var_label(df$s2)

  # Set labels from dictionary, e.g. as read from external file
  # One description is missing, one has no match
  description = tibble(
    name = c(
      "Sepal.Length", "Sepal.Width", "Petal.Length", "Petal.Width",
      "Something"),
    label = c(
      "Sepal length", "Sepal width",  "Petal length", "Petal width",
      "something")
  )
  var_labels <- setNames(as.list(description$label), description$name)
  iris_labelled <- iris \%>\%
    set_variable_labels(.labels = var_labels, .strict = FALSE)
  var_label(iris_labelled)

  # defining variable labels derived from variable names
  if (require(snakecase)) {
    iris <- iris \%>\%
      set_variable_labels(.labels = to_sentence_case(names(iris)))
    var_label(iris)
  }

  # example with a vector
  v <- 1:5
  v <- v \%>\% set_variable_labels("a variable label")
  v
  v \%>\% set_variable_labels(NULL)
}
}
