/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 
#define H5B2_TESTING    

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 

typedef struct H5B2_test_ctx_t {
    uint8_t sizeof_size; 
} H5B2_test_ctx_t;

static void  *H5B2__test_crt_context(void *udata);
static herr_t H5B2__test_dst_context(void *ctx);
static herr_t H5B2__test_store(void *nrecord, const void *udata);
static herr_t H5B2__test_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5B2__test_encode(uint8_t *raw, const void *nrecord, void *ctx);
static herr_t H5B2__test_decode(const uint8_t *raw, void *nrecord, void *ctx);
static herr_t H5B2__test_debug(FILE *stream, int indent, int fwidth, const void *record, const void *_udata);

static herr_t H5B2__test2_store(void *nrecord, const void *udata);
static herr_t H5B2__test2_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5B2__test2_encode(uint8_t *raw, const void *nrecord, void *ctx);
static herr_t H5B2__test2_decode(const uint8_t *raw, void *nrecord, void *ctx);
static herr_t H5B2__test2_debug(FILE *stream, int indent, int fwidth, const void *record, const void *_udata);

const H5B2_class_t H5B2_TEST[1] = {{
    
    H5B2_TEST_ID,           
    "H5B2_TEST_ID",         
    sizeof(hsize_t),        
    H5B2__test_crt_context, 
    H5B2__test_dst_context, 
    H5B2__test_store,       
    H5B2__test_compare,     
    H5B2__test_encode,      
    H5B2__test_decode,      
    H5B2__test_debug        
}};

const H5B2_class_t H5B2_TEST2[1] = {{
    
    H5B2_TEST2_ID,           
    "H5B2_TEST2_ID",         
    sizeof(H5B2_test_rec_t), 
    H5B2__test_crt_context,  
    H5B2__test_dst_context,  
    H5B2__test2_store,       
    H5B2__test2_compare,     
    H5B2__test2_encode,      
    H5B2__test2_decode,      
    H5B2__test2_debug        
}};

H5FL_DEFINE_STATIC(H5B2_test_ctx_t);

static void *
H5B2__test_crt_context(void *_f)
{
    H5F_t           *f = (H5F_t *)_f;  
    H5B2_test_ctx_t *ctx;              
    void            *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (ctx = H5FL_MALLOC(H5B2_test_ctx_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "can't allocate callback context");

    
    ctx->sizeof_size = H5F_SIZEOF_SIZE(f);

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__test_dst_context(void *_ctx)
{
    H5B2_test_ctx_t *ctx = (H5B2_test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    ctx = H5FL_FREE(H5B2_test_ctx_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(hsize_t *)nrecord = *(const hsize_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test_compare(const void *rec1, const void *rec2, int *result)
{
    FUNC_ENTER_PACKAGE_NOERR

    *result = (int)(*(const hssize_t *)rec1 - *(const hssize_t *)rec2);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test_encode(uint8_t *raw, const void *nrecord, void *_ctx)
{
    H5B2_test_ctx_t *ctx = (H5B2_test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    H5_ENCODE_LENGTH_LEN(raw, *(const hsize_t *)nrecord, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test_decode(const uint8_t *raw, void *nrecord, void *_ctx)
{
    H5B2_test_ctx_t *ctx = (H5B2_test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    H5_DECODE_LENGTH_LEN(raw, *(hsize_t *)nrecord, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test_debug(FILE *stream, int indent, int fwidth, const void *record, const void H5_ATTR_UNUSED *_udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(record);

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth, "Record:", *(const hsize_t *)record);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test2_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5B2_test_rec_t *)nrecord = *(const H5B2_test_rec_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test2_compare(const void *rec1, const void *rec2, int *result)
{
    FUNC_ENTER_PACKAGE_NOERR

    *result = (int)(((const H5B2_test_rec_t *)rec1)->key - ((const H5B2_test_rec_t *)rec2)->key);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test2_encode(uint8_t *raw, const void *nrecord, void *_ctx)
{
    H5B2_test_ctx_t *ctx = (H5B2_test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    H5_ENCODE_LENGTH_LEN(raw, ((const H5B2_test_rec_t *)nrecord)->key, ctx->sizeof_size);
    H5_ENCODE_LENGTH_LEN(raw, ((const H5B2_test_rec_t *)nrecord)->val, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test2_decode(const uint8_t *raw, void *nrecord, void *_ctx)
{
    H5B2_test_ctx_t *ctx = (H5B2_test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    H5_DECODE_LENGTH_LEN(raw, ((H5B2_test_rec_t *)nrecord)->key, ctx->sizeof_size);
    H5_DECODE_LENGTH_LEN(raw, ((H5B2_test_rec_t *)nrecord)->val, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B2__test2_debug(FILE *stream, int indent, int fwidth, const void *record, const void H5_ATTR_UNUSED *_udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(record);

    Rfprintf(stream, "%*s%-*s (%" PRIuHSIZE ", %" PRIuHSIZE ")\n", indent, "", fwidth,
            "Record:", ((const H5B2_test_rec_t *)record)->key, ((const H5B2_test_rec_t *)record)->val);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5B2__get_root_addr_test(H5B2_t *bt2, haddr_t *root_addr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(bt2);
    assert(root_addr);

    
    *root_addr = bt2->hdr->root.addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5B2__get_node_info_test(H5B2_t *bt2, void *udata, H5B2_node_info_test_t *ninfo)
{
    H5B2_hdr_t     *hdr;                 
    H5B2_node_ptr_t curr_node_ptr;       
    void           *parent = NULL;       
    uint16_t        depth;               
    int             cmp;                 
    unsigned        idx;                 
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(bt2);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    curr_node_ptr = hdr->root;

    
    if (hdr->swmr_write)
        parent = hdr;

    
    depth = hdr->depth;

    
    if (0 == curr_node_ptr.node_nrec)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree has no records");

    
    cmp = -1;
    while (depth > 0 && cmp != 0) {
        H5B2_internal_t *internal;      
        H5B2_node_ptr_t  next_node_ptr; 

        
        if (NULL == (internal = H5B2__protect_internal(hdr, parent, &curr_node_ptr, depth, false,
                                                       H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree internal node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");

        if (cmp > 0)
            idx++;

        if (cmp != 0) {
            
            next_node_ptr = internal->node_ptrs[idx];

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                               (unsigned)(hdr->swmr_write ? H5AC__PIN_ENTRY_FLAG : H5AC__NO_FLAGS_SET)) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            if (hdr->swmr_write)
                parent = internal;

            
            curr_node_ptr = next_node_ptr;
        } 
        else {
            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            ninfo->depth = depth;
            ninfo->nrec  = curr_node_ptr.node_nrec;

            
            HGOTO_DONE(SUCCEED);
        } 

        
        depth--;
    } 

    {
        H5B2_leaf_t *leaf; 

        
        if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, &curr_node_ptr, false, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

        
        if (cmp != 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "record not in B-tree");
    } 

    
    ninfo->depth = depth;
    ninfo->nrec  = curr_node_ptr.node_nrec;

done:
    if (parent) {
        assert(ret_value < 0);
        if (parent != hdr && H5AC_unpin_entry(parent) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5B2__get_node_depth_test(H5B2_t *bt2, void *udata)
{
    H5B2_node_info_test_t ninfo;          
    int                   ret_value = -1; 

    FUNC_ENTER_PACKAGE

    
    assert(bt2);

    
    if (H5B2__get_node_info_test(bt2, udata, &ninfo) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, (-1), "error looking up node info");

    
    ret_value = (int)ninfo.depth;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
