% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/basepcor.R
\docType{class}
\name{basepcor-class}
\alias{basepcor-class}
\alias{basepcor}
\alias{basepcor.numeric}
\alias{basepcor.matrix}
\alias{print.basepcor}
\title{Information for a base model for correlation matrices}
\usage{
basepcor(base, p, itheta, d0)

\method{basepcor}{numeric}(base, p, itheta, d0)

\method{basepcor}{matrix}(base, p, itheta, d0)

\method{print}{basepcor}(x, ...)
}
\arguments{
\item{base}{matrix (or numeric vector) as the base correlation
(or parameter at the base model).}

\item{p}{integer (needed if \code{base} is vector): the dimention.}

\item{itheta}{integer vector or 'graphpcor' to specify the (vectorized)
position where 'theta' is placed in the initial (before the fill-in)
Cholesky (lower triangle) factor. If missing, default, assumes
the dense case as \code{itheta = which(lower.tri(...))}, giving
\code{length(theta)=p(p-1)/2}.}

\item{d0}{numeric vector to specify the diagonal of the
Cholesky factor for the initial precision matrix \code{Q0}.
Default, if not provided, is \code{d0 = p:1}.}

\item{x}{a basepcor object.}

\item{...}{further arguments passed on.}
}
\value{
a basepcor object

a \code{basepcor} object
}
\description{
The \code{basepcor} class contain a correlation matrix \code{base},
the parameter vector \code{theta}, that generates
or is generated by \code{base}, the dimension \code{p},
the index \code{itheta} for \code{theta} in the (lower) Cholesky,
and the Hessian around it \code{I0}, see details.
}
\details{
The Inverse Transform Parametrization - ITP,
is applied by starting with a
\deqn{\mathbf{L}^{(0)} = \left[ \begin{array}{ccccc}
\textrm{p} & 0 & 0 & \ldots & 0 \\
\theta_1 & \textrm{p-}1 & 0 & \ldots & 0 \\
\theta_2 & \theta_p & \textrm{p-}2 & \ddots & \vdots \\
\vdots & \vdots & \ddots & \ddots & 0 \\
\theta_{p-1} & \theta_{2p-3} & \ldots & \theta_m & 1
\end{array}\right] .}

Then compute \eqn{\mathbf{Q}^{(0)}} =
\eqn{\mathbf{L}\mathbf{L}^{T}},
\eqn{\mathbf{V}^{(0)}} = \eqn{(\mathbf{Q}^{(0)})^{-1}} and
\eqn{s_{i}^{(0)}} = \eqn{\sqrt{\mathbf{V}_{i,i}^{(0)}}}, and
define \eqn{\mathbf{S}^{(0)}} = diag\eqn{(s_1^{(0)},\ldots,s_p^{(0)})}
in order to have \eqn{\mathbf{C}}=
\eqn{\mathbf{S}^{-1}\mathbf{V}^{(0)}\mathbf{S}^{-1}}.

The decomposition of the Hessian matrix around the base model,
\code{I0}, formally \eqn{\mathbf{I}(\theta_0)}, is numerically computed.
This element has the following attributes:
'h.5' as \eqn{\mathbf{I}^{1/2}(\theta_0)}, and
'hneg.5' as \eqn{\mathbf{I}^{-1/2}(\theta_0)}.
}
\section{Functions}{
\itemize{
\item \code{basepcor()}: Build a \code{basepcor} object.

\item \code{basepcor(numeric)}: Build a \code{basepcor} from the parameter vector.

\item \code{basepcor(matrix)}: Build a \code{basepcor} from a correlation matrix.

\item \code{print(basepcor)}: Print method for 'basepcor'

}}
\examples{

## p = 3, m = 2
bc <- basepcor(c(-1,-1), p = 3, itheta = c(2,3))
bc

round(solve(bc$base), 4)

all.equal(bc, basepcor(bc$base, itheta =c(2,3)))

## p = 4, m = 4
th2 <- c(0.5,-1,0.5,-0.3)
ith2 <- c(2,3,8,12)
b2 <- basepcor(th2, p = 4, itheta = ith2)
b2

Sparse(solve(b2$base), zeros.rm = TRUE)

all.equal(th2, basepcor(b2$base, itheta = ith2)$theta)

## Hessian around the base (and its decomposition, etc.)
b2$I0
}
