#' Information for a base model for correlation matrices
#' @description
#' The `basepcor` class contain a correlation matrix `base`,
#' the parameter vector `theta`, that generates
#' or is generated by `base`, the dimension `p`,
#' the index `itheta` for `theta` in the (lower) Cholesky,
#' and the Hessian around it `I0`, see details.
setClass(
  "basepcor",
  slots = c("base", "theta", "p", "I0", "itheta"),
  validity = function(object) {
    (object$p>1) &&
      (object$p == nrow(object$base)) &&
      all.equal(object$base == t(object$base)) &&
      all(diag(object$base==1)) &&
      all(diag(chol(object$base))>0)
  }
)
#' @describeIn basepcor
#' Build a `basepcor` object.
#' @param base matrix (or numeric vector) as the base correlation
#' (or parameter at the base model).
#' @param p integer (needed if `base` is vector): the dimention.
#' @param itheta integer vector or 'graphpcor' to specify the (vectorized)
#' position where 'theta' is placed in the initial (before the fill-in)
#' Cholesky (lower triangle) factor. If missing, default, assumes
#' the dense case as `itheta = which(lower.tri(...))`, giving
#' `length(theta)=p(p-1)/2`.
#' @param d0 numeric vector to specify the diagonal of the
#' Cholesky factor for the initial precision matrix `Q0`.
#' Default, if not provided, is `d0 = p:1`.
#' @details
#' The Inverse Transform Parametrization - ITP,
#' is applied by starting with a
#' \deqn{\mathbf{L}^{(0)} = \left[ \begin{array}{ccccc}
#' \textrm{p} & 0 & 0 & \ldots & 0 \\
#' \theta_1 & \textrm{p-}1 & 0 & \ldots & 0 \\
#' \theta_2 & \theta_p & \textrm{p-}2 & \ddots & \vdots \\
#' \vdots & \vdots & \ddots & \ddots & 0 \\
#' \theta_{p-1} & \theta_{2p-3} & \ldots & \theta_m & 1
#' \end{array}\right] .}
#'
#' Then compute \eqn{\mathbf{Q}^{(0)}} =
#' \eqn{\mathbf{L}\mathbf{L}^{T}},
#' \eqn{\mathbf{V}^{(0)}} = \eqn{(\mathbf{Q}^{(0)})^{-1}} and
#' \eqn{s_{i}^{(0)}} = \eqn{\sqrt{\mathbf{V}_{i,i}^{(0)}}}, and
#' define \eqn{\mathbf{S}^{(0)}} = diag\eqn{(s_1^{(0)},\ldots,s_p^{(0)})}
#'  in order to have \eqn{\mathbf{C}}=
#'  \eqn{\mathbf{S}^{-1}\mathbf{V}^{(0)}\mathbf{S}^{-1}}.
#'
#' The decomposition of the Hessian matrix around the base model,
#' `I0`, formally \eqn{\mathbf{I}(\theta_0)}, is numerically computed.
#' This element has the following attributes:
#' 'h.5' as \eqn{\mathbf{I}^{1/2}(\theta_0)}, and
#' 'hneg.5' as \eqn{\mathbf{I}^{-1/2}(\theta_0)}.
#' @returns a basepcor object
#' @export
basepcor <- function(
    base,
    p,
    itheta,
    d0) {
  UseMethod("basepcor")
}
#' @describeIn basepcor
#' Build a `basepcor` from the parameter vector.
#' @returns a `basepcor` object
#' @export
#' @example demo/basepcor.R
basepcor.numeric <- function(
    base,
    p,
    itheta,
    d0) {
  theta <- base
  m <- length(theta)
  if(missing(p)) {
    if(missing(itheta)) {
      p <- (1 + sqrt(1+8*m))/2
      stopifnot(floor(p)==ceiling(p))
    } else {
      stop("please provid 'p'!")
    }
  }
  stopifnot(p>1)
  if(missing(itheta)) {
    itheta <- which(lower.tri(diag(
      x = rep(1, p), nrow = p, ncol = p)))
  } else {
    if(inherits(itheta, "graphpcor")) {
      Q1 <- Laplacian(itheta)
      itheta <- which(lower.tri(Q1) & (Q1 != 0.0))
    }
  }
  stopifnot(length(itheta) == m)
  if(missing(d0)) {
    d0 <- p:1
  }

  ## compute L0
  L <- Lprec0(
    theta = theta,
    p = p,
    itheta = itheta,
    d0 = d0
  )
  lq02cor <- function(l) {
    cov2cor(chol2inv(t(l)))
  }

  ## compute the base correlation matrix
  base <- lq02cor(L)
  U0correl <- chol(base)

  ## Hessian around theta
  I0 <- hessian(
    func = function(x)
      KLD10(C1 = lq02cor(Lprec0(x, p = p, itheta = itheta, d0 = d0)),
            L0 = U0correl),
    x = theta)

  ## eigen decomposition of I0, I0^0.5 and I0^-0.5
  eI0 <- eigen(I0)
  stopifnot(all(eI0$values>0))
  s <- sqrt(eI0$values)
  tv <- t(eI0$vectors)
  attr(I0, "decomposition") <- eI0
  attr(I0, "determinant") <- prod(eI0$values)
  attr(I0, "h.5") <- crossprod(tv * s, tv)
  attr(I0, "hneg.5") <- crossprod(tv / s, tv)

  ## output
  out <- list(
    base = base,
    theta = theta,
    p = p,
    d0 = d0,
    itheta = itheta,
    I0 = I0)
  class(out) <- "basepcor"

  return(out)
}

#' @describeIn basepcor
#' Build a `basepcor` from a correlation matrix.
#' @export
basepcor.matrix <- function(
    base,
    p,
    itheta,
    d0) {
  stopifnot(all.equal(base, t(base)))
  p <- as.integer(nrow(base))
  if(missing(d0)) {
    d0 <- p:1
  }
  stopifnot((length(d0)==p) && (all(d0>0)))
  U0correl <- chol(base)
  Q <- chol2inv(U0correl)
  ilQ <-  which(
    lower.tri(matrix(1, p, p)) &
      (!is.zero(Q, tol = 1e-9)))
  if(missing(itheta)) {
    itheta <- ilQ
  }  else {
    if(inherits(itheta, "graphpcor")) {
      Q1 <- Laplacian(itheta)
      itheta <- which(lower.tri(Q1) & (Q1 != 0.0))
    }
  }
  stopifnot(all(ilQ %in% itheta))

  ## compute theta
  LQ0 <- t(chol(Q))
  for(i in 1:p) {
      LQ0[i, ] <- (d0[i]/LQ0[i, i]) * LQ0[i, ]
  }
  theta <- LQ0[itheta]

  ## Hessian around theta
  lq02cor <- function(l) {
    cov2cor(chol2inv(t(l)))
  }
  I0 <- hessian(
    func = function(x)
      KLD10(C1 = lq02cor(Lprec0(x, p = p, itheta = itheta, d0 = d0)),
            L0 = U0correl),
    x = theta)

  ## eigen decomposition of I0, I0^0.5 and I0^-0.5
  eI0 <- eigen(I0)
  stopifnot(all(eI0$values>0))
  s <- sqrt(eI0$values)
  tv <- t(eI0$vectors)
  attr(I0, "decomposition") <- eI0
  attr(I0, "determinant") <- prod(eI0$values)
  attr(I0, "h.5") <- crossprod(tv * s, tv)
  attr(I0, "hneg.5") <- crossprod(tv / s, tv)

  ## output
  out <- list(
    base = base,
    theta = theta,
    p = p,
    d0 = d0,
    itheta = itheta,
    I0 = I0)
  class(out) <- "basepcor"

  return(out)
}
#' @describeIn basepcor
#' Print method for 'basepcor'
#' @param x a basepcor object.
#' @param ... further arguments passed on.
#' @export
print.basepcor <- function(x, ...) {
  cat("Parameters:\n", sep = "")
  print(x$theta, ...)
  cat("Base correlation matrix:\n")
  print(x$base, ...)
}
