% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggaverage.R, R/ggpredict.R
\name{ggaverage}
\alias{ggaverage}
\alias{ame}
\alias{ggpredict}
\alias{mem}
\title{Get marginal effects from model terms}
\usage{
ggaverage(model, terms, ci.lvl = 0.95, type = c("fe", "re"),
  typical = "mean", ppd = FALSE, ...)

ame(model, terms, ci.lvl = 0.95, type = c("fe", "re"), typical = "mean",
  ppd = FALSE, ...)

ggpredict(model, terms, ci.lvl = 0.95, type = c("fe", "re"),
  full.data = FALSE, typical = "mean", ppd = FALSE, x.as.factor = FALSE,
  pretty = TRUE, ...)

mem(model, terms, ci.lvl = 0.95, type = c("fe", "re"), full.data = FALSE,
  typical = "mean", ppd = FALSE, x.as.factor = FALSE, ...)
}
\arguments{
\item{model}{A fitted model object, or a list of model objects. Any model
that supports common methods like \code{predict()}, \code{family()}
or \code{model.frame()} should work.}

\item{terms}{Character vector with the names of those terms from \code{model},
for which marginal effects should be displayed. At least one term
is required to calculate effects for certain terms, maximum length is
three terms, where the second and third term indicate the groups, i.e.
predictions of first term are grouped by the levels of the second (and third)
term. If \code{terms} is missing or \code{NULL}, marginal effects for each
model term are calculated. Indicating levels in square brackets allows for
selecting only specific groups. Term name and levels in brackets must be
separated by a whitespace character, e.g.
\code{terms = c("age", "education [1,3]")}. Numeric ranges, separated
with colon, are also allowed: \code{terms = c("education", "age [30:60]")}.
Furthermore, it is possible to specify a function name. Values for
predictions will then be transformed, e.g. \code{terms = "income [exp]"}.
This is useful when model predictors were transformed for fitting the
model and should be back-transformed to the original scale for predictions.
See 'Examples' and package vignettes. All remaining covariates that are
not specified in \code{terms} are held constant (if \code{full.data = FALSE},
the default) or are set to the values from the observations (i.e. are kept
as they happen to be; see 'Details').}

\item{ci.lvl}{Numeric, the level of the confidence intervals. For \code{ggpredict()},
use \code{ci.lvl = NA}, if confidence intervals should not be calculated
(for instance, due to computation time).}

\item{type}{Character, only applies for mixed effects models. Indicates
whether predicted values should be conditioned on random effects
(\code{type = "re"}) or fixed effects only (\code{type = "fe"},
the default). If \code{type = "re"}, the uncertainty in the variance
parameters is currently not taken into account. The specific levels of
the random effects are not provided in the \code{newdata}-argument. Hence,
if \code{type = "re"}, \code{predict()} is called with \code{re.form = NULL},
i.e. predictions are conditioned on random effects only using the first level
of random effects, but the uncertainty in the variance parameters is ignored.}

\item{typical}{Character vector, naming the function to be applied to the
covariates over which the effect is "averaged". The default is "mean".
See \code{\link[sjstats]{typical_value}} for options.}

\item{ppd}{Logical, if \code{TRUE}, predictions for Stan-models are
based on the posterior predictive distribution
(\code{\link[rstantools]{posterior_predict}}). If \code{FALSE} (the
default), predictions are based on posterior draws of the linear
predictor (\code{\link[rstantools]{posterior_linpred}}).}

\item{...}{Further arguments passed down to \code{predict()}.}

\item{full.data}{Logical, if \code{TRUE}, the returned data frame contains
predictions for all observations. This data frame also has columns
for residuals and observed values, and can also be used to plot a
scatter plot of all data points or fitted values.
If \code{FALSE} (the default), the returned data frame only contains
predictions for all combinations of unique values of the model's
predictors. Residuals and observed values are set to \code{NA}.
Usually, this argument is only used internally by \code{ggaverage()}.}

\item{x.as.factor}{Logical, if \code{TRUE}, preserves factor-class as
\code{x}-column in the returned data frame. By default, the \code{x}-column
is always numeric.}

\item{pretty}{Logical, if \code{TRUE}, terms with many unique values (more
than 25 distinct values) will be "prettyfied" using the \code{pretty()}-function.
The number of intervals is the square-root of the value range of the
specific term, which will lead to reduced unique values for which predictions
need to be calculated. This is especially useful in cases where
out-of-memory-errors may occur, or if predictions should be computed for
representative pretty values. \code{pretty = FALSE} calculates predictions
for all values of continuous variables in \code{terms}, even if these
terms have many unique values. This is useful, for example, for splines.}
}
\value{
A tibble (with \code{ggeffects} class attribute) with consistent data columns:
        \describe{
          \item{\code{x}}{the values of the first term in \code{terms}, used as x-position in plots.}
          \item{\code{predicted}}{the predicted values of the response, used as y-position in plots.}
          \item{\code{conf.low}}{the lower bound of the confidence interval for the predicted values.}
          \item{\code{conf.high}}{the upper bound of the confidence interval for the predicted values.}
          \item{\code{observed}}{if \code{full.data = TRUE}, this columns contains the observed values (the response vector).}
          \item{\code{residuals}}{if \code{full.data = TRUE}, this columns contains residuals.}
          \item{\code{group}}{the grouping level from the second term in \code{terms}, used as grouping-aesthetics in plots.}
          \item{\code{facet}}{the grouping level from the third term in \code{terms}, used to indicate facets in plots.}
        }
        For proportional odds logistic regression (see \code{\link[MASS]{polr}})
        resp. cumulative link models (e.g., see \code{\link[ordinal]{clm}}),
        an additional column \code{response.level} is returned, which indicates
        the grouping of predictions based on the level of the model's response.
}
\description{
\code{ggpredict()} computes predicted (fitted) values for the
  response, at the margin of specific values from certain model terms,
  where additional model terms indicate the grouping structure.
  \code{ggaverage()} computes the average predicted values.
  The result is returned as tidy data frame.
  \cr \cr
  \code{mem()} is an alias for \code{ggpredict()} (marginal effects
  at the mean), \code{ame()} is an alias for \code{ggaverage()}
  (average marginal effects).
}
\details{
Currently supported model-objects are: \code{lm, glm, glm.nb, lme, lmer,
  glmer, glmer.nb, nlmer, glmmTMB, gam, vgam, gamm, gamm4, multinom,
  betareg, gls, gee, plm, lrm, polr, clm, hurdle, zeroinfl, svyglm,
  svyglm.nb, truncreg, coxph, stanreg, brmsfit}.
  Other models not listed here are passed to a generic predict-function
  and might work as well, or maybe with \code{ggeffect()}, which
  effectively does the same as \code{ggpredict()}. The main difference
  between \code{ggpredict()} and \code{ggeffect()} is how factors are
  held constant: \code{ggpredict()} uses the reference level, while
  \code{ggeffect()} computes a kind of "average" value, which represents
  the proportions of each factor's category.
  \cr \cr
  For \code{ggpredict()}, if \code{full.data = FALSE}, \code{expand.grid()}
  is called on all unique combinations of \code{model.frame(model)[, terms]}
  and used as \code{newdata}-argument for \code{predict()}. In this case,
  all remaining covariates that are not specified in \code{terms} are
  held constant. Numeric values are set to the mean (unless changed
  with the \code{typical}-argument), factors are set to their
  reference level and character vectors to their mode (most common
  element).
  \cr \cr
  \code{ggaverage()} computes the average predicted values, by calling
  \code{ggpredict()} with \code{full.data = TRUE}, where argument
  \code{newdata = model.frame(model)} is used in \code{predict()}.
  Hence, predictions are made on the model data. In this case, all
  remaining covariates that are not specified in \code{terms} are
  \emph{not} held constant, but vary between observations (and are
  kept as they happen to be). The predicted values are then averaged
  for each group (if any).
  \cr \cr
  Thus, \code{ggpredict()} can be considered as calculating marginal
  effects at the mean, while \code{ggaverage()} computes average
  marginal effects.
  \cr \cr
  \code{ggpredict()} also works with \strong{Stan}-models from
  the \CRANpkg{rstanarm} or \CRANpkg{brms}-package. The predicted
  values are the median value of all drawn posterior samples. The
  confidence intervals for Stan-models are actually high density
  intervals, computed by \code{\link[sjstats]{hdi}}, unless \code{ppd = TRUE}.
  If \code{ppd = TRUE}, predictions are based on draws of the posterior
  predictive  distribution and the uncertainty interval is computed
  using  \code{\link[rstantools]{predictive_interval}}. By default (i.e.
  \code{ppd = FALSE}), the predictions are based on
  \code{\link[rstantools]{posterior_linpred}} and hence have some
  limitations: the uncertainty of the error term is not taken into
  account. The recommendation is to use the posterior predictive
  distribution (\code{\link[rstantools]{posterior_predict}}).
  Note that for binomial models, the \code{newdata}-argument
  used in \code{posterior_predict()} must also contain the vector
  with the number of trials. In this case, a dummy-vector is used,
  where all values for the response are set to 1.
}
\note{
Since data for \code{ggaverage()} comes from the model frame, not all
  possible combinations of values in \code{terms} might be present in the data,
  thus lines or confidence bands from \code{plot()} might not span over
  the complete x-axis-range.
  \cr \cr
  There are some limitations for certain model objects. For example,
  it is currently only possible to compute predicted risk scores for
  \code{coxph}-models, but not expected number of events nor survival
  probabilities.
  \cr \cr
  \code{polr}- or \code{clm}-models have an additional column
  \code{response.level}, which indicates with which level of the response
  variable the predicted values are associated.
}
\examples{
data(efc)
fit <- lm(barthtot ~ c12hour + neg_c_7 + c161sex + c172code, data = efc)

ggpredict(fit, terms = "c12hour")
ggpredict(fit, terms = "c12hour", full.data = TRUE)
ggpredict(fit, terms = c("c12hour", "c172code"))
ggpredict(fit, terms = c("c12hour", "c172code", "c161sex"))

# only range of 40 to 60 for variable 'c12hour'
ggpredict(fit, terms = "c12hour [40:60]")

# to plot ggeffects-objects, you can use the 'plot()'-function.
# the following examples show how to build your ggplot by hand.

# plot predicted values, remaining covariates held constant
library(ggplot2)
mydf <- ggpredict(fit, terms = "c12hour")
ggplot(mydf, aes(x, predicted)) +
  geom_line() +
  geom_ribbon(aes(ymin = conf.low, ymax = conf.high), alpha = .1)

# with "full.data = TRUE", remaining covariates vary between
# observations, so fitted values can be plotted
mydf <- ggpredict(fit, terms = "c12hour", full.data = TRUE)
ggplot(mydf, aes(x, predicted)) + geom_point()

# you can add a smoothing-geom to show the linear trend of fitted values
ggplot(mydf, aes(x, predicted)) +
  geom_smooth(method = "lm", se = FALSE) +
  geom_point()

# three variables, so we can use facets and groups
mydf <- ggpredict(
  fit,
  terms = c("c12hour", "c161sex", "c172code"),
  full.data = TRUE
)
ggplot(mydf, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE) +
  facet_wrap(~facet, ncol = 2)

# average marginal effects
mydf <- ggaverage(fit, terms = c("c12hour", "c172code"))
ggplot(mydf, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE)

# select specific levels for grouping terms
mydf <- ggpredict(fit, terms = c("c12hour", "c172code [1,3]", "c161sex"))
ggplot(mydf, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE) +
  facet_wrap(~facet) +
  labs(
    y = get_y_title(mydf),
    x = get_x_title(mydf),
    colour = get_legend_title(mydf)
  )

# level indication also works for factors with non-numeric levels
# and in combination with numeric levels for other variables
library(sjlabelled)
data(efc)
efc$c172code <- as_label(efc$c172code)
fit <- lm(barthtot ~ c12hour + neg_c_7 + c161sex + c172code, data = efc)
ggpredict(fit, terms = c("c12hour",
  "c172code [low level of education, high level of education]",
  "c161sex [1]"))

# use categorical value on x-axis, use axis-labels, add error bars
dat <- ggpredict(fit, terms = c("c172code", "c161sex"))
ggplot(dat, aes(x, predicted, colour = group)) +
  geom_point(position = position_dodge(.1)) +
  geom_errorbar(
    aes(ymin = conf.low, ymax = conf.high),
    position = position_dodge(.1)
  ) +
  scale_x_continuous(breaks = 1:3, labels = get_x_labels(dat))

# 3-way-interaction with 2 continuous variables
data(efc)
# make categorical
efc$c161sex <- as_factor(efc$c161sex)
fit <- lm(neg_c_7 ~ c12hour * barthtot * c161sex, data = efc)
# select only levels 30, 50 and 70 from continuous variable Barthel-Index
dat <- ggpredict(fit, terms = c("c12hour", "barthtot [30,50,70]", "c161sex"))
ggplot(dat, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE, fullrange = TRUE) +
  facet_wrap(~facet) +
  labs(
    colour = get_legend_title(dat),
    x = get_x_title(dat),
    y = get_y_title(dat),
    title = get_title(dat)
  )

# or with ggeffects' plot-method
\dontrun{
plot(dat, ci = FALSE)}

# use factor levels as x-column in returned data frame
data(efc)
efc$c161sex <- as_label(efc$c161sex)
fit <- lm(neg_c_7 ~ c12hour + c161sex, data = efc)
ggpredict(fit, terms = "c161sex", x.as.factor = TRUE)

}
