% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/acmcp.R
\name{acmcp}
\alias{acmcp}
\alias{print.acmcp}
\alias{summary.acmcp}
\alias{print.summary.acmcp}
\title{Autocorrelated multistep-ahead conformal prediction method}
\usage{
acmcp(
  object,
  alpha = 1 - 0.01 * object$level,
  ncal = 10,
  rolling = FALSE,
  integrate = TRUE,
  scorecast = TRUE,
  lr = 0.1,
  Tg = NULL,
  delta = NULL,
  Csat = 2/pi * (ceiling(log(Tg) * delta) - 1/log(Tg)),
  KI = max(abs(object$errors), na.rm = TRUE),
  ...
)
}
\arguments{
\item{object}{An object of class \code{"cvforecast"}. It must have an argument
\code{x} for original univariate time series, an argument \code{MEAN} for
point forecasts and \code{ERROR} for forecast errors on validation set.
See the results of a call to \code{\link{cvforecast}}.}

\item{alpha}{A numeric vector of significance levels to achieve a desired
coverage level \eqn{1-\alpha}.}

\item{ncal}{Length of the burn-in period for training the scorecaster.
If \code{rolling = TRUE}, it is also used as the length of the trailing windows
for learning rate calculation and the windows for the calibration set.
If \code{rolling = FALSE}, it is used as the initial period of calibration sets
and trailing windows for learning rate calculation.}

\item{rolling}{If \code{TRUE}, a rolling window strategy will be adopted to
form the trailing window for learning rate calculation and the calibration set
for scorecaster if applicable. Otherwise, expanding window strategy will be used.}

\item{integrate}{If \code{TRUE}, error integration will be included in the
update process.}

\item{scorecast}{If \code{TRUE}, scorecasting will be included in the update
process.}

\item{lr}{Initial learning rate used for quantile tracking.}

\item{Tg}{The time that is set to achieve the target absolute coverage
guarantee before this.}

\item{delta}{The target absolute coverage guarantee is set to \eqn{1-\alpha-\delta}.}

\item{Csat}{A positive constant ensuring that by time \code{Tg}, an absolute
guarantee is of at least \eqn{1-\alpha-\delta} coverage.}

\item{KI}{A positive constant to place the integrator on the same scale as the scores.}

\item{...}{Other arguments are passed to the function.}
}
\value{
A list of class \code{c("acmcp", "cpforecast", "forecast")}
with the following components:
\item{x}{The original time series.}
\item{series}{The name of the series \code{x}.}
\item{method}{A character string "acmcp".}
\item{cp_times}{The number of times the conformal prediction is performed in
cross-validation.}
\item{MEAN}{Point forecasts as a multivariate time series, where the \eqn{h}th column
holds the point forecasts for forecast horizon \eqn{h}. The time index
corresponds to the period for which the forecast is produced.}
\item{ERROR}{Forecast errors given by
\eqn{e_{t+h|t} = y_{t+h}-\hat{y}_{t+h|t}}{e[t+h] = y[t+h]-f[t+h]}.}
\item{LOWER}{A list containing lower bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{UPPER}{A list containing upper bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{level}{The confidence values associated with the prediction intervals.}
\item{call}{The matched call.}
\item{model}{A list containing information abouth the conformal prediction model.}
If \code{mean} is included in the \code{object}, the components \code{mean},
\code{lower}, and \code{upper} will also be returned, showing the information
about the test set forecasts generated using all available observations.
}
\description{
Compute prediction intervals and other information by applying the Autocorrelated
Multistep-ahead Conformal Prediction (AcMCP) method. The method can only
deal with asymmetric nonconformity scores, i.e., forecast errors.
}
\details{
Similar to the PID method, the AcMCP method also integrates three modules (P, I, and D) to
form the final iteration. However, instead of performing conformal prediction
for each individual forecast horizon \code{h} separately, AcMCP employs a combination
of an MA\eqn{(h-1)} model and a linear regression model of \eqn{e_{t+h|t}} on
\eqn{e_{t+h-1|t},\dots,e_{t+1|t}} as the scorecaster. This allows the AcMCP method
to capture the relationship between the \eqn{h}-step ahead forecast error and
past errors.
}
\examples{
# Simulate time series from an AR(2) model
library(forecast)
series <- arima.sim(n = 200, list(ar = c(0.8, -0.5)), sd = sqrt(1))

# Cross-validation forecasting
far2 <- function(x, h, level) {
  Arima(x, order = c(2, 0, 0)) |>
    forecast(h = h, level)
}
fc <- cvforecast(series, forecastfun = far2, h = 3, level = 95,
                 forward = TRUE, initial = 1, window = 50)

# AcMCP setup
Tg <- 200; delta <- 0.01
Csat <- 2 / pi * (ceiling(log(Tg) * delta) - 1 / log(Tg))
KI <- 2
lr <- 0.1

# AcMCP with integrator and scorecaster
acmcpfc <- acmcp(fc, ncal = 50, rolling = TRUE,
             integrate = TRUE, scorecast = TRUE,
             lr = lr, KI = KI, Csat = Csat)
print(acmcpfc)
summary(acmcpfc)

}
\references{
Wang, X., and Hyndman, R. J. (2024). "Online conformal inference
for multi-step time series forecasting", arXiv preprint arXiv:2410.13115.
}
\seealso{
\code{\link{pid}}
}
