#' simulate_l
#' 
#' @description
#' 
#' This function generates simulated data using a
#' user-defined objective function. You can specify the number of
#' iterations to control how many data are generated.
#' These datasets can be used for parameter recovery and model recovery. 
#' For more information, please refer to the GitHub repository:
#' https://github.com/yuki-961004/binaryRL
#' 
#' @param data [data.frame] raw data. 
#'  This data should include the following mandatory columns: 
#'  - "sub", "time_line", "L_choice", "R_choice", "L_reward", "R_reward". 
#'  
#' @param id [vector] which subject is going to be analyzed.
#'  is being analyzed. The value should correspond to an entry in the "sub" 
#'  column, which must contain the subject IDs. 
#'  e.g., `id = c(1:40)`
#'  
#' @param obj_func [function] a function with only ONE argument `params`. 
#'  Additionally, it is important to note that the data needs to be retrieved 
#'  from parent.frame(). This function returns the binaryRL.res(res).
#' 
#' @param n_params [integer] The number of free parameters in your model. 
#' 
#' @param n_trials [integer] The total number of trials in your experiment.
#' 
#' @param lower [vector] lower bounds of free parameters
#' 
#' @param upper [vector] upper bounds of free parameters
#' 
#' @param seed [integer] random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  default: `seed = 123` 
#'  
#' @param iteration [integer] the number of iteration
#'
#' @returns a list with fake data generated by random free parameters
#' @export
#'
simulate_list <- function(
  data,
  id = 1,
  obj_func, 
  n_params, 
  n_trials,
  lower, 
  upper,
  iteration = 10,
  seed = 123
) {
  list_simulated <- list()
  # 检测是都用同一个被试的题目, 还是每次都更换题目
  if (length(id) == 1) {
    id <- rep(id, iteration)
  }
  
  for (i in 1:iteration) {
    params <- c()
    
    for (j in 1:n_params) {
      # 确保每次种子不同
      set.seed(seed + n_params * i + j) 
      if (j == n_params) {
        params[j] <- stats::rexp(1, rate = upper[j]) + lower[j]
      } else {
        # 其他参数服从均匀分布
        params[j] <- stats::runif(n = 1, min = lower[j], max = upper[j])
      }
    }
    
    # 创建临时环境
    binaryRL.env <- new.env()
    mode <- "simulate"
    # 将data传入到临时环境
    assign(x = "mode", value = mode, envir = binaryRL.env)
    assign(x = "data", value = data, envir = binaryRL.env)
    assign(x = "id", value = id[i], envir = binaryRL.env)
    assign(x = "n_params", value = n_params, envir = binaryRL.env)
    assign(x = "n_trials", value = n_trials, envir = binaryRL.env)
    # 让obj_func的环境绑定在fit_env中
    environment(obj_func) <- binaryRL.env
    
    list_simulated[[i]] <- obj_func(params = params)
    list_simulated[[i]]$input <- params
  }
  
  return(list_simulated)
}
