% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OTC.R
\name{OTC}
\alias{OTC}
\title{Find the optimal testing configuration}
\usage{
OTC(algorithm, p = NULL, probabilities = NULL, Se = 0.99, Sp = 0.99,
  group.sz, obj.fn = c("ET", "MAR"), weights = NULL, alpha = 2)
}
\arguments{
\item{algorithm}{character string defining the group testing algorithm to be used.
Non-informative testing options include two-stage hierarchical ("\kbd{D2}"),
three-stage hierarchical ("\kbd{D3}"), square array testing without master 
pooling ("\kbd{A2}"), and square array testing without master pooling ("\kbd{A2M}"). 
Informative testing options include two-stage hierarchical ("\kbd{ID2}"),
three-stage hierarchical ("\kbd{ID3}"), and square array testing without 
master pooling ("\kbd{IA2}").}

\item{p}{overall probability of disease that will be used to generate a
vector/matrix of individual probabilities. For non-informative algorithms, a 
homogeneous set of probabilities will be used. For informative algorithms, the 
\code{\link{p.vec.func}} function will be used to generate a heterogeneous 
set of probabilities. Either \kbd{p} or \kbd{probabilities} should be specified, 
but not both.}

\item{probabilities}{a vector of individual probabilities, which is homogeneous 
for non-informative testing algorithms and heterogeneous for informative 
testing algorithms. Either  \kbd{p} or \kbd{probabilities} should be specified, 
but not both.}

\item{Se}{the sensitivity of the diagnostic test.}

\item{Sp}{the specificity of the diagnostic test.}

\item{group.sz}{a single group size or range of group sizes for which to 
calculate operating characteristics and/or find the OTC. The details of group 
size specification are given under 'Details'.}

\item{obj.fn}{a list of objective functions which are minimized to find the
OTC. The expected number of tests per individual, "\kbd{ET}", will always 
be calculated. Additional options include "\kbd{MAR}" 
(the expected number of tests divided by the expected number of correct 
classifications, described in Malinovsky et al. (2016)), and "\kbd{GR}" 
(a linear combination of the expected number of tests, the number of 
misclassified negatives, and the number of misclassified positives, 
described in Graff & Roeloffs (1972)). See Hitt et al. (2018) at
\url{http://chrisbilder.com/grouptesting} for additional details.}

\item{weights}{a matrix of up to six sets of weights for the GR function. 
Each set of weights is specified by a row of the matrix.}

\item{alpha}{a shape parameter for the beta distribution that specifies the degree of
heterogeneity for the generated probability vector (for informative testing only).}
}
\value{
A list containing:
\item{prob}{the probability of disease, as specified by the user.}
\item{alpha}{level of heterogeneity for the generated probability vector
(for informative testing only).}
\item{Se}{the sensitivity of the diagnostic test.}
\item{Sp}{the specificity of the diagnostic test.}
\item{opt.ET, opt.MAR, opt.GR}{a list for each objective function specified
by the user, containing:
\describe{
\item{OTC}{a list specifying elements of the optimal testing configuration, 
which may include:
\describe{
\item{Stage1}{pool size for the first stage of hierarchical testing, if applicable.}
\item{Stage2}{pool sizes for the second stage of hierarchical testing, if applicable.}
\item{Block.sz}{the block size/initial group size for informative Dorfman testing,
which is not tested.}
\item{pool.szs}{pool sizes for the first stage of testing for informative Dorfman
testing.}
\item{Array.dim}{the row/column size for array testing.}
\item{Array.sz}{the overall array size for array testing (the square of the row/column size).}}}
\item{p.vec}{the sorted vector of individual probabilities, if applicable.}
\item{p.mat}{the sorted matrix of individual probabilities in gradient arrangement,
if applicable.}
\item{ET}{the expected testing expenditure for the OTC.}
\item{value}{the value of the objective function per individual.}
\item{PSe}{the overall pooling sensitivity for the algorithm. Further details 
are given under 'Details'.}
\item{PSp}{the overall pooling specificity for the algorithm. Further details 
are given under 'Details'.}
\item{PPPV}{the overall pooling positive predictive value for the algorithm. 
Further details are given under 'Details'.}
\item{PNPV}{the overall pooling negative predictive value for the algorithm. 
Further details are given under 'Details'.}}}
}
\description{
Find the optimal testing configuration (OTC) for standard group testing
algorithms and calculate the associated operating characteristics.
}
\details{
This function finds the OTC and computes the
associated operating characteristics for standard group testing algorithms,
as described in Hitt et al. (2018) at 
\url{http://chrisbilder.com/grouptesting}.

Available algorithms include two- and three-stage hierarchical testing and
array testing with and without master pooling. Both non-informative and informative
group testing settings are allowed for each algorithm, except informative 
array testing with master pooling is unavailable because this method has not 
appeared in the group testing literature. Operating characteristics calculated are
expected number of tests, pooling sensitivity, pooling specificity, pooling
positive predictive value, and pooling negative predictive value for each individual.

The value(s) specified by \kbd{group.sz} represent the initial (stage 1) 
group size for three-stage hierarchical testing and non-informative 
two-stage hierarchical testing. For informative two-stage hierarchical testing, 
the \kbd{group.sz} specified represents the block size used in the pool-specific
optimal Dorfman (PSOD) method, where the initial group (block) is not
tested. For more details on informative two-stage hierarchical testing 
implemented via the PSOD method, see Hitt et al. (2018) and McMahan et al. (2012a).
For array testing without master pooling, the \kbd{group.sz} specified
represents the row/column size for initial (stage 1) testing. For array testing 
with master pooling, the \kbd{group.sz} specified represents the row/column size 
for stage 2 testing. The group size for initial testing is overall array size, 
given by the square of the row/column size.

If a single value is provided for \kbd{group.sz} with array testing or
non-informative two-stage hierarchical testing, operating
characteristics will be calculated and no optimization will be performed.
If a single value is provided for \kbd{group.sz} with three-stage hierarchical or 
informative two-stage hierarchical, the OTC will be  
found over all possible configurations. If a range of group sizes is specified, 
the OTC will be found over all group sizes.

The displayed pooling sensitivity, pooling specificity, pooling positive 
predictive value, and pooling negative predictive value are weighted 
averages of the corresponding individual accuracy measures for all 
individuals within the initial group for a hierarchical algorithm, or 
within the entire array for an array-based algorithm.
Expressions for these averages are provided in the Supplementary 
Material for Hitt et al. (2018). These expressions are based on accuracy 
definitions given by Altman and Bland (1994a, 1994b).
}
\examples{
# Find the OTC for non-informative
#   two-stage hierarchical (Dorfman) testing
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
OTC(algorithm="D2", p=0.05, Se=0.99, Sp=0.99, group.sz=2:100,
obj.fn=c("ET", "MAR"))

# Find the OTC for informative
#   two-stage hierarchical (Dorfman) testing, implemented
#   via the pool-specific optimal Dorfman (PSOD) method
#   described in McMahan et al. (2012a), where the greedy
#   algorithm proposed for PSOD is replaced by considering
#   all possible testing configurations.
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta 
#   distribution with p = 0.01 and a heterogeneity level 
#   of alpha = 0.5. Depending on the specified probability, 
#   alpha level, and overall group size, simulation may 
#   be necessary in order to generate the vector of individual
#   probabilities. This is done using p.vec.func() and 
#   requires the user to set a seed in order to reproduce 
#   results.
# This example takes approximately 2.5 minutes to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
\dontrun{
set.seed(52613)
OTC(algorithm="ID2", p=0.01, Se=0.95, Sp=0.95, group.sz=50,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 0.5, 0.5), 
nrow=3, ncol=2, byrow=TRUE), alpha=0.5)}

# Find the OTC over all possible
#   testing configurations for a specified group size for
#   non-informative three-stage hierarchical testing
# This example takes approximately 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
OTC(algorithm="D3", p=0.001, Se=0.95, Sp=0.95, group.sz=18,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1), nrow=1, ncol=2, byrow=TRUE))

# Find the OTC for non-informative
#   three-stage hierarchical testing
# This example takes approximately 20 seconds to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
\dontrun{
OTC(algorithm="D3", p=0.06, Se=0.90, Sp=0.90, 
group.sz=3:30, obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 100, 100), 
nrow=3, ncol=2, byrow=TRUE))}

# Find the OTC over all possible configurations 
#   for a specified group size, given a 
#   heterogeneous vector of probabilities.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
OTC(algorithm="ID3", probabilities=c(0.012, 0.014, 0.011, 
0.012, 0.010, 0.015), Se=0.99, Sp=0.99, group.sz=6, 
obj.fn=c("ET","MAR","GR"), weights=matrix(data=c(1, 1), 
nrow=1, ncol=2, byrow=TRUE), alpha=0.5)

# Calculate the operating characteristics for a specified array size
#   for non-informative array testing without master pooling
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
OTC(algorithm="A2", p=0.005, Se=0.95, Sp=0.95, group.sz=8, 
obj.fn=c("ET", "MAR"))

# Find the OTC for informative array testing without 
#   master pooling
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta 
#   distribution with p = 0.03 and a heterogeneity level 
#   of alpha = 2. The probabilities are then arranged in 
#   a matrix using the gradient method described in 
#   McMahan et al. (2012b). Depending on the specified 
#   probability, alpha level, and overall group size, 
#   simulation may be necessary in order to generate the 
#   vector of individual probabilities. This is done using 
#   p.vec.func() and requires the user to set a 
#   seed in order to reproduce results.
# This example takes approximately 30 seconds to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
\dontrun{
set.seed(1002)
OTC(algorithm="IA2", p=0.03, Se=0.95, Sp=0.95, 
group.sz=3:20, obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 100, 100), 
nrow=3, ncol=2, byrow=TRUE), alpha=2)}

# Find the OTC for non-informative array testing 
#   with master pooling
# This example takes approximately 20 seconds to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
\dontrun{
OTC(algorithm="A2M", p=0.02, Se=0.90, Sp=0.90, 
group.sz=3:20, obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 0.5, 0.5, 2, 2, 
100, 100, 10, 100), nrow=6, ncol=2, byrow=TRUE))}
}
\references{
\insertRef{Altman1994a}{binGroup}

\insertRef{Altman1994b}{binGroup}

\insertRef{Graff1972}{binGroup}

\insertRef{Hitt2018}{binGroup}

\insertRef{Malinovsky2016}{binGroup}

\insertRef{McMahan2012a}{binGroup}

\insertRef{McMahan2012b}{binGroup}
}
\seealso{
\code{\link{NI.Dorf}} for non-informative two-stage (Dorfman) testing, \code{\link{Inf.Dorf}} for
informative two-stage (Dorfman) testing, \code{\link{NI.D3}} for non-informative three-stage
hierarchical testing, \code{\link{Inf.D3}} for informative three-stage hierarchical testing,
\code{\link{NI.Array}} for non-informative array testing, \code{\link{Inf.Array}} for informative
array testing, and \code{\link{NI.A2M}} for non-informative array testing with master pooling.

\url{http://chrisbilder.com/grouptesting}

Other OTC functions: \code{\link{Inf.Array}},
  \code{\link{Inf.D3}}, \code{\link{Inf.Dorf}},
  \code{\link{NI.A2M}}, \code{\link{NI.Array}},
  \code{\link{NI.D3}}, \code{\link{NI.Dorf}}
}
\author{
Brianna D. Hitt
}
