% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/p_significance.R
\name{p_significance}
\alias{p_significance}
\alias{p_significance.numeric}
\alias{p_significance.get_predicted}
\alias{p_significance.data.frame}
\alias{p_significance.brmsfit}
\title{Practical Significance (ps)}
\usage{
p_significance(x, ...)

\method{p_significance}{numeric}(x, threshold = "default", ...)

\method{p_significance}{get_predicted}(
  x,
  threshold = "default",
  use_iterations = FALSE,
  verbose = TRUE,
  ...
)

\method{p_significance}{data.frame}(x, threshold = "default", rvar_col = NULL, ...)

\method{p_significance}{brmsfit}(
  x,
  threshold = "default",
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{threshold}{The threshold value that separates significant from
negligible effect, which can have following possible values:
\itemize{
\item \code{"default"}, in which case the range is set to \code{0.1} if input is a vector,
and based on \code{\link[=rope_range]{rope_range()}} if a (Bayesian) model is provided.
\item a single numeric value (e.g., 0.1), which is used as range around zero
(i.e. the threshold range is set to -0.1 and 0.1, i.e. reflects a symmetric
interval)
\item a numeric vector of length two (e.g., \code{c(-0.2, 0.1)}), useful for
asymmetric intervals
\item a list of numeric vectors, where each vector corresponds to a parameter
\item a list of \emph{named} numeric vectors, where names correspond to parameter
names. In this case, all parameters that have no matching name in \code{threshold}
will be set to \code{"default"}.
}}

\item{use_iterations}{Logical, if \code{TRUE} and \code{x} is a \code{get_predicted} object,
(returned by \code{\link[insight:get_predicted]{insight::get_predicted()}}), the function is applied to the
iterations instead of the predictions. This only applies to models that return
iterations for predicted values (e.g., \code{brmsfit} models).}

\item{verbose}{Toggle off warnings.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should variables for fixed effects (\code{"fixed"}), random effects
(\code{"random"}) or both (\code{"all"}) be returned? Only applies to mixed models. May
be abbreviated.

For models of from packages \strong{brms} or \strong{rstanarm} there are additional
options:
\itemize{
\item \code{"fixed"} returns fixed effects.
\item \code{"random_variance"} return random effects parameters (variance and
correlation components, e.g. those parameters that start with \code{sd_} or
\code{cor_}).
\item \code{"grouplevel"} returns random effects group level estimates, i.e. those
parameters that start with \code{r_}.
\item \code{"random"} returns both \code{"random_variance"} and \code{"grouplevel"}.
\item \code{"all"} returns fixed effects and random effects variances.
\item \code{"full"} returns all parameters.
}}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
Values between 0 and 1 corresponding to the probability of practical significance (ps).
}
\description{
Compute the probability of \strong{Practical Significance} (\emph{\strong{ps}}), which can
be conceptualized as a unidirectional equivalence test. It returns the
probability that effect is above a given threshold corresponding to a
negligible effect in the median's direction. Mathematically, it is defined as
the proportion of the posterior distribution of the median sign above the
threshold.
}
\details{
\code{p_significance()} returns the proportion of a probability
distribution (\code{x}) that is outside a certain range (the negligible
effect, or ROPE, see argument \code{threshold}). If there are values of the
distribution both below and above the ROPE, \code{p_significance()} returns
the higher probability of a value being outside the ROPE. Typically, this
value should be larger than 0.5 to indicate practical significance. However,
if the range of the negligible effect is rather large compared to the
range of the probability distribution \code{x}, \code{p_significance()}
will be less than 0.5, which indicates no clear practical significance.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (require("rstanarm")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

# Simulate a posterior distribution of mean 1 and SD 1
# ----------------------------------------------------
posterior <- rnorm(1000, mean = 1, sd = 1)
p_significance(posterior)

# Simulate a dataframe of posterior distributions
# -----------------------------------------------
df <- data.frame(replicate(4, rnorm(100)))
p_significance(df)
\donttest{
# rstanarm models
# -----------------------------------------------
model <- rstanarm::stan_glm(mpg ~ wt + cyl,
  data = mtcars,
  chains = 2, refresh = 0
)
p_significance(model)
# multiple thresholds - asymmetric, symmetric, default
p_significance(model, threshold = list(c(-10, 5), 0.2, "default"))
# named thresholds
p_significance(model, threshold = list(wt = 0.2, `(Intercept)` = c(-10, 5)))
}
\dontshow{\}) # examplesIf}
}
