% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/p_direction.R
\name{p_direction}
\alias{p_direction}
\alias{pd}
\alias{p_direction.numeric}
\alias{p_direction.data.frame}
\alias{p_direction.brmsfit}
\alias{p_direction.get_predicted}
\title{Probability of Direction (pd)}
\usage{
p_direction(x, ...)

pd(x, ...)

\method{p_direction}{numeric}(
  x,
  method = "direct",
  null = 0,
  as_p = FALSE,
  remove_na = TRUE,
  ...
)

\method{p_direction}{data.frame}(
  x,
  method = "direct",
  null = 0,
  as_p = FALSE,
  remove_na = TRUE,
  rvar_col = NULL,
  ...
)

\method{p_direction}{brmsfit}(
  x,
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  method = "direct",
  null = 0,
  as_p = FALSE,
  remove_na = TRUE,
  ...
)

\method{p_direction}{get_predicted}(
  x,
  method = "direct",
  null = 0,
  as_p = FALSE,
  remove_na = TRUE,
  use_iterations = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{A vector representing a posterior distribution, a data frame of
posterior draws (samples be parameter). Can also be a Bayesian model.}

\item{...}{Currently not used.}

\item{method}{Can be \code{"direct"} or one of methods of \code{\link[=estimate_density]{estimate_density()}},
such as \code{"kernel"}, \code{"logspline"} or \code{"KernSmooth"}. See details.}

\item{null}{The value considered as a "null" effect. Traditionally 0, but
could also be 1 in the case of ratios of change (OR, IRR, ...).}

\item{as_p}{If \code{TRUE}, the p-direction (pd) values are converted to a
frequentist p-value using \code{\link[=pd_to_p]{pd_to_p()}}.}

\item{remove_na}{Should missing values be removed before computation? Note
that \code{Inf} (infinity) are \emph{not} removed.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should results for fixed effects (\code{"fixed"}, the default),
random effects (\code{"random"}) or both ("\verb{all"}) be returned? Only applies to
mixed models. May be abbreviated.}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{use_iterations}{Logical, if \code{TRUE} and \code{x} is a \code{get_predicted} object,
(returned by \code{\link[insight:get_predicted]{insight::get_predicted()}}), the function is applied to the
iterations instead of the predictions. This only applies to models that return
iterations for predicted values (e.g., \code{brmsfit} models).}

\item{verbose}{Toggle off warnings.}
}
\value{
Values between 0.5 and 1 \emph{or} between 0 and 1 (see above) corresponding to
the probability of direction (pd).
}
\description{
Compute the \strong{Probability of Direction} (\emph{\strong{pd}}, also known as the Maximum
Probability of Effect - \emph{MPE}). This can be interpreted as the probability
that a parameter (described by its posterior distribution) is strictly
positive or negative (whichever is the most probable). Although differently
expressed, this index is fairly similar (\emph{i.e.}, is strongly correlated) to
the frequentist \strong{p-value} (see details).
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{What is the \emph{pd}?}{


The Probability of Direction (pd) is an index of effect existence, representing
the certainty with which an effect goes in a particular direction (i.e., is
positive or negative / has a sign), typically ranging from 0.5 to 1 (but see
next section for cases where it can range between 0 and 1). Beyond
its simplicity of interpretation, understanding and computation, this index
also presents other interesting properties:
\itemize{
\item Like other posterior-based indices, \emph{pd} is solely based on the posterior
distributions and does not require any additional information from the data
or the model (e.g., such as priors, as in the case of Bayes factors).
\item It is robust to the scale of both the response variable and the predictors.
\item It is strongly correlated with the frequentist p-value, and can thus
be used to draw parallels and give some reference to readers non-familiar
with Bayesian statistics (Makowski et al., 2019).
}
}

\section{Relationship with the p-value}{


In most cases, it seems that the \emph{pd} has a direct correspondence with the
frequentist one-sided \emph{p}-value through the formula (for two-sided \emph{p}):
\ifelse{html}{\out{p = 2 * (1 - p<sub>d</sub>)}}{\eqn{p = 2 \times (1 - p_d)}}
Thus, a two-sided p-value of respectively \code{.1}, \code{.05}, \code{.01} and \code{.001} would
correspond approximately to a \emph{pd} of \verb{95\%}, \verb{97.5\%}, \verb{99.5\%} and \verb{99.95\%}.
See \code{\link[=pd_to_p]{pd_to_p()}} for details.
}

\section{Possible Range of Values}{


The largest value \emph{pd} can take is 1 - the posterior is strictly directional.
However, the smallest value \emph{pd} can take depends on the parameter space
represented by the posterior.

\strong{For a continuous parameter space}, exact values of 0 (or any point null
value) are not possible, and so 100\% of the posterior has \emph{some} sign, some
positive, some negative. Therefore, the smallest the \emph{pd} can be is 0.5 -
with an equal posterior mass of positive and negative values. Values close to
0.5 \emph{cannot} be used to support the null hypothesis (that the parameter does
\emph{not} have a direction) is a similar why to how large p-values cannot be used
to support the null hypothesis (see \code{\link[=pd_to_p]{pd_to_p()}}; Makowski et al., 2019).

\strong{For a discrete parameter space or a parameter space that is a mixture
between discrete and continuous spaces}, exact values of 0 (or any point
null value) \emph{are} possible! Therefore, the smallest the \emph{pd} can be is 0 -
with 100\% of the posterior mass on 0. Thus values close to 0 can be used to
support the null hypothesis (see van den Bergh et al., 2021).

Examples of posteriors representing discrete parameter space:
\itemize{
\item When a parameter can only take discrete values.
\item When a mixture prior/posterior is used (such as the spike-and-slab prior;
see van den Bergh et al., 2021).
\item When conducting Bayesian model averaging (e.g., \code{\link[=weighted_posteriors]{weighted_posteriors()}} or
\code{brms::posterior_average}).
}
}

\section{Methods of computation}{


The \emph{pd} is defined as:
\deqn{p_d = max({Pr(\hat{\theta} < \theta_{null}), Pr(\hat{\theta} > \theta_{null})})}{pd = max(mean(x < null), mean(x > null))}

The most simple and direct way to compute the \emph{pd} is to compute the
proportion of positive (or larger than \code{null}) posterior samples, the
proportion of negative (or smaller than \code{null}) posterior samples, and take
the larger of the two. This "simple" method is the most straightforward, but
its precision is directly tied to the number of posterior draws.

The second approach relies on \link[=estimate_density]{density estimation}: It starts by
estimating the continuous-smooth density function (for which many methods are
available), and then computing the \link[=area_under_curve]{area under the curve}
(AUC) of the density curve on either side of \code{null} and taking the maximum
between them. Note the this approach assumes a continuous density function,
and so \strong{when the posterior represents a (partially) discrete parameter
space, only the direct method \emph{must} be used} (see above).
}

\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (requireNamespace("rstanarm", quietly = TRUE) && requireNamespace("emmeans", quietly = TRUE) && requireNamespace("brms", quietly = TRUE) && requireNamespace("BayesFactor", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

# Simulate a posterior distribution of mean 1 and SD 1
# ----------------------------------------------------
posterior <- rnorm(1000, mean = 1, sd = 1)
p_direction(posterior)
p_direction(posterior, method = "kernel")

# Simulate a dataframe of posterior distributions
# -----------------------------------------------
df <- data.frame(replicate(4, rnorm(100)))
p_direction(df)
p_direction(df, method = "kernel")

\donttest{
# rstanarm models
# -----------------------------------------------
model <- rstanarm::stan_glm(mpg ~ wt + cyl,
  data = mtcars,
  chains = 2, refresh = 0
)
p_direction(model)
p_direction(model, method = "kernel")

# emmeans
# -----------------------------------------------
p_direction(emmeans::emtrends(model, ~1, "wt", data = mtcars))

# brms models
# -----------------------------------------------
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
p_direction(model)
p_direction(model, method = "kernel")

# BayesFactor objects
# -----------------------------------------------
bf <- BayesFactor::ttestBF(x = rnorm(100, 1, 1))
p_direction(bf)
p_direction(bf, method = "kernel")
}
\dontshow{\}) # examplesIf}
\dontshow{if (requireNamespace("posterior", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Using "rvar_col"
x <- data.frame(mu = c(0, 0.5, 1), sigma = c(1, 0.5, 0.25))
x$my_rvar <- posterior::rvar_rng(rnorm, 3, mean = x$mu, sd = x$sigma)
x
p_direction(x, rvar_col = "my_rvar")
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Makowski, D., Ben-Shachar, M. S., Chen, S. A., & Lüdecke, D. (2019).
Indices of effect existence and significance in the Bayesian framework.
Frontiers in psychology, 10, 2767. \doi{10.3389/fpsyg.2019.02767}
\item van den Bergh, D., Haaf, J. M., Ly, A., Rouder, J. N., & Wagenmakers, E. J.
(2021). A cautionary note on estimating effect size. Advances in Methods
and Practices in Psychological Science, 4(1). \doi{10.1177/2515245921992035}
}
}
\seealso{
\code{\link[=pd_to_p]{pd_to_p()}} to convert between Probability of Direction (pd) and p-value.
}
