% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bdplyr.R
\name{bd_write}
\alias{bd_write}
\alias{bd_write_rds}
\alias{bd_write_csv}
\title{Writes the result of operations with \code{\link[=bdplyr]{bdplyr()}} to disk}
\usage{
bd_write(.lazy_tbl, .write_fn, path, overwrite = FALSE, ...)

bd_write_rds(.lazy_tbl, path, overwrite = FALSE, compress = "none", ...)

bd_write_csv(.lazy_tbl, path, overwrite = FALSE, ...)
}
\arguments{
\item{.lazy_tbl}{A lazy tibble, tipically the output of \code{\link[=bdplyr]{bdplyr()}}.}

\item{.write_fn}{A function for writing the result of a tibble to
disk. Do not use () afther the function's name, the function \emph{object} should be passed. Some functions the user might consider are:
\link[writexl:write_xlsx]{writexl::write_xlsx}, \link[jsonlite:read_json]{jsonlite::write_json}, \link[foreign:write.dta]{foreign::write.dta},
\link[arrow:write_feather]{arrow::write_feather}, etc.}

\item{path}{String containing the path for the file to be created.
The desired folders must already exist and the file should normally end with
the corresponding extension.}

\item{overwrite}{FALSE by default. Indicates whether the local file should
be overwritten if it already exists. Use with care.}

\item{...}{Parameters passed to the \code{.write_fn} function.}

\item{compress}{For \code{\link[=bd_write_rds]{bd_write_rds()}} only. Compression method to use: "none"
(default), "gz" ,"bz", or "xz", in ascending order of compression.
Remember that the higher the compression, the smaller the file size on disk,
ut also the longer the time to load the data. See also: \code{\link[readr:read_rds]{readr::write_rds()}}.}
}
\value{
String containing the path to the created file.
}
\description{
Writes a remote table to disk that was called via {bdplyr}.
It will collect the data and write to disk in the chosen format.
You will only need this function if you have not yet collected the data
using the \code{\link[=bd_collect]{bd_collect()}}.

The comprehensive function \code{\link[=bd_write]{bd_write()}} takes as a parameter \code{.write_fn},
which will be the name of some function (without parentheses) capable of
writing a tibble to disk.

As helpers, the \code{\link[=bd_write_rds]{bd_write_rds()}} and \code{\link[=bd_write_csv]{bd_write_csv()}} functions make it
easier to write in these formats, more common in everyday life, calling
writing functions from \code{{readr}} package.
}
\examples{
\dontrun{

 cool_db <- basedosdados::

# setup billing
basedosdados::set_billing_id("MY-BILLING-ID")

# connect with a Base dos Dados db

cool_db_ssp <- basedosdados::bdplyr(
 "basedosdados.br_sp_gov_ssp.ocorrencias_registradas")

# subset the data
my_subset <- cool_db_ssp \%>\%
 dplyr::filter(ano == 2021, mes == 04)

# write it in csv - generic function

basedosdados::bd_write(.lazy_tbl = my_subset,
                      .write_fn = write.csv,
                      "data-raw/ssp_subset.csv"
)

# write in .xlsx
basedosdados::bd_write(.lazy_tbl = my_subset,
                      .write_fn = writexl::write_xlsx,
                      "data-raw/ssp_subset.xlsx"
)

# using the derivatives functions
# to csv
basedosdados::bd_write_csv(.lazy_tbl = my_subset,
                          "data-raw/ssp_subset2.csv"
)

#' # to rds
basedosdados::bd_write_rds(.lazy_tbl = my_subset,
                           "data-raw/ssp_subset.rds"
)

# to rds - with compression
basedosdados::bd_write_rds(.lazy_tbl = my_subset,
                           "data-raw/ssp_subset2.rds",
                           compress = "gz"
)

# to rds - with HARD compression
basedosdados::bd_write_rds(.lazy_tbl = my_subset,
                           "data-raw/ssp_subset3.rds",
                           compress = "xz"
)

## using other write functions

# json
basedosdados::bd_write(.lazy_tbl = my_subset,
                       .write_fn = jsonlite::write_json,
                       "data-raw/ssp_subset.json"
)

# dta
basedosdados::bd_write(.lazy_tbl = my_subset,
                       .write_fn = foreign::write.dta,
                       "data-raw/ssp_subset.dta")
)

# feather
basedosdados::bd_write(.lazy_tbl = my_subset,
                       .write_fn = arrow::write_feather,
                       "data-raw/ssp_subset.feather"
)
}
}
