% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotSPC.R
\docType{methods}
\name{plotSPC}
\alias{plotSPC}
\alias{plot,SoilProfileCollection,ANY-method,plot.SoilProfileCollection}
\alias{plot}
\alias{plot,SoilProfileCollection,ANY-method}
\title{Create Soil Profile Sketches}
\usage{
plotSPC(
  x,
  color = "soil_color",
  width = 0.2,
  name = NULL,
  name.style = "right-center",
  label = idname(x),
  hz.depths = FALSE,
  alt.label = NULL,
  alt.label.col = "black",
  cex.names = 0.5,
  cex.depth.axis = cex.names,
  cex.id = cex.names + (0.2 * cex.names),
  font.id = 2,
  print.id = TRUE,
  id.style = "auto",
  plot.order = 1:length(x),
  relative.pos = 1:length(x),
  add = FALSE,
  scaling.factor = 1,
  y.offset = 0,
  x.idx.offset = 0,
  n = length(x),
  max.depth = ifelse(is.infinite(max(x)), 200, max(x)),
  n.depth.ticks = 5,
  shrink = FALSE,
  shrink.cutoff = 3,
  abbr = FALSE,
  abbr.cutoff = 5,
  divide.hz = TRUE,
  hz.distinctness.offset = NULL,
  hz.topography.offset = NULL,
  hz.boundary.lty = NULL,
  axis.line.offset = -2.5,
  plot.depth.axis = TRUE,
  density = NULL,
  col.label = color,
  col.palette = c("#5E4FA2", "#3288BD", "#66C2A5", "#ABDDA4", "#E6F598", "#FEE08B",
    "#FDAE61", "#F46D43", "#D53E4F", "#9E0142"),
  col.palette.bias = 1,
  col.legend.cex = 1,
  n.legend = 8,
  lwd = 1,
  lty = 1,
  default.color = grey(0.95),
  ...
)

## S4 method for signature 'SoilProfileCollection,ANY'
## note: y argument in generic definition is not currently used
plot(x, y, ...)
}
\arguments{
\item{x}{a \code{SoilProfileCollection} object}

\item{color}{quoted column name containing R-compatible color descriptions, or numeric / categorical data to be displayed thematically; see details}

\item{width}{scaling of profile widths (typically 0.1 - 0.4)}

\item{name}{quoted column name of the (horizon-level) attribute containing horizon designations, can be left as \code{NULL} and horizon designation column will be selected via \code{hzdesgnname(x)}. Suppress horizon name printing by setting \code{name=NA}.}

\item{name.style}{one of several possible horizon designations labeling styles: 'right-center' (aqp default), 'left-top', 'left-center'}

\item{label}{quoted column name of the (site-level) attribute used to identify profile sketches}

\item{hz.depths}{logical, annotate horizon top depths to the right of each sketch (FALSE)}

\item{alt.label}{quoted column name of the (site-level) attribute used for secondary annotation}

\item{alt.label.col}{color used for secondary annotation text}

\item{cex.names}{baseline character scaling applied to all text labels}

\item{cex.depth.axis}{character scaling applied to depth scale}

\item{cex.id}{character scaling applied to \code{label}}

\item{font.id}{font style applied to \code{label}, default is 2 (bold)}

\item{print.id}{logical, print \code{label} above/beside each profile? (TRUE)}

\item{id.style}{\code{label} printing style: 'auto' (default) = simple heuristic used to select from: 'top' = centered above each profile, 'side' = 'along the top-left edge of profiles'}

\item{plot.order}{integer vector describing the order in which individual soil profiles should be plotted}

\item{relative.pos}{vector of relative positions along the x-axis, within \{1, n\}, ignores \code{plot.order} see details}

\item{add}{logical, add to an existing figure}

\item{scaling.factor}{vertical scaling of profile depths, useful for adding profiles to an existing figure}

\item{y.offset}{vertical offset for top of profiles, useful for adding profiles to an existing figure}

\item{x.idx.offset}{integer specifying horizontal offset from 0 (left-hand edge)}

\item{n}{integer describing amount of space along x-axis to allocate, defaults to \code{length(x)}}

\item{max.depth}{suggested lower depth boundary of plot}

\item{n.depth.ticks}{suggested number of ticks in depth scale}

\item{shrink}{logical, reduce character scaling for 'long' horizon by 80\%?}

\item{shrink.cutoff}{character length defining 'long' horizon names}

\item{abbr}{logical, abbreviate \code{label}?}

\item{abbr.cutoff}{suggested minimum length for abbreviated \code{label}}

\item{divide.hz}{logical, divide horizons with line segment? (TRUE), see details}

\item{hz.distinctness.offset}{NULL, or quoted column name (horizon-level attribute) containing vertical offsets used to depict horizon boundary distinctness (same units as profiles), see details and code{\link{hzDistinctnessCodeToOffset}}}

\item{hz.topography.offset}{NULL, or quoted column name (horizon-level attribute) containing offsets used to depict horizon boundary topography (same units as profiles), see details and code{\link{hzTopographyCodeToOffset}}}

\item{hz.boundary.lty}{quoted column name (horizon-level attribute) containing line style (integers) used to encode horizon topography}

\item{axis.line.offset}{horizontal offset applied to depth axis (default is -2.5, larger numbers move the axis to the right)}

\item{plot.depth.axis}{logical, plot depth axis? (default is TRUE)}

\item{density}{fill density used for horizon color shading, either a single integer or a quoted column name (horizon-level attribute) containing integer values (default is NULL, no shading)}

\item{col.label}{thematic legend title}

\item{col.palette}{color palette used for thematic sketches (default is \code{rev(brewer.pal(10, 'Spectral'))})}

\item{col.palette.bias}{color ramp bias (skew), see \code{\link{colorRamp}}}

\item{col.legend.cex}{scaling of thematic legend}

\item{n.legend}{approximate number of classes used in numeric legend, max number of items per row in categorical legend}

\item{lwd}{line width multiplier used for sketches}

\item{lty}{line style used for sketches}

\item{default.color}{default horizon fill color used when \code{color} attribute is \code{NA}}

\item{\dots}{other arguments passed into lower level plotting functions}

\item{y}{(not used)}
}
\description{
Generate a diagram of soil profile sketches from a \code{SoilProfileCollection} object. The \href{https://ncss-tech.github.io/AQP/aqp/aqp-intro.html}{Introduction to SoilProfileCollection Objects tutorial} contains many examples and discussion of the large number of arguments to this function.
}
\details{
Depth limits (\code{max.depth}) and number of depth ticks (\code{n.depth.ticks}) are \emph{suggestions} to the \code{\link{pretty}} function. You may have to tinker with both parameters to get what you want.

The 'side' \code{id.style} is useful when plotting a large collection of profiles, and/or, when profile IDs are long.

If the column containing horizon designations is not specified (the \code{name} argument), a column (presumed to contain horizon designation labels) is guessed based on regular expression matching of the pattern 'name'-- this usually works, but it is best to manual specify the name of the column containing horizon designations.

The \code{color} argument can either name a column containing R-compatible colors, possibly created via \code{\link{munsell2rgb}}, or column containing either numeric or categorical (either factor or character) values. In the second case, values are converted into colors and displayed along with a simple legend above the plot. Note that this functionality makes several assumptions about plot geometry and is most useful in an interactive setting.

Adjustments to the legend can be specified via \code{col.label} (legend title), \code{col.palette} (palette of colors, automatically expanded), \code{col.legend.cex} (legend scaling), and \code{n.legend} (approximate number of classes for numeric variables, or, maximum number of legend items per row for categorical variables). Currently, \code{plotSPC} will only generate two rows of legend items. Consider reducing the number of classes if two rows isn't enough room.

Profile sketches can be added according to relative positions along the x-axis (vs. integer sequence) via \code{relative.pos} argument. This should be a vector of positions within \{1,n\} that are used for horizontal placement. Default values are \code{1:length(x)}. Care must be taken when both \code{plot.order} and \code{relative.pos} are used simultaneously: \code{relative.pos} specifies horizontal placement after sorting. \code{addDiagnosticBracket} and \code{addVolumeFraction} use the \code{relative.pos} values for subsequent annotation.

Relative positions that are too close will result in overplotting of sketches. Adjustments to relative positions such that overlap is minimized can be performed with \code{fixOverlap(pos)}, where \code{pos} is the original vector of relative positions.

The \code{x.idx.offset} argument can be used to shift a collection of pedons from left to right in the figure. This can be useful when plotting several different \code{SoilProfileCollection} objects within the same figure. Space must be pre-allocated in the first plotting call, with an offset specified in the second call. See examples below.
}
\note{
A new plot of soil profiles is generated, or optionally added to an existing plot.
}
\examples{

# example data
data(sp1)
# usually best to adjust margins
par(mar=c(0,0,3,0))

# add color vector
sp1$soil_color <- with(sp1, munsell2rgb(hue, value, chroma))

# promote to SoilProfileCollection
depths(sp1) <- id ~ top + bottom

# plot profiles
plot(sp1, id.style='side')

# title, note line argument:
title('Sample Data 1', line=1, cex.main=0.75)

# plot profiles without horizon-line divisions
plot(sp1, divide.hz=FALSE)

# add dashed lines illustrating horizon boundary distinctness
sp1$hzD <- hzDistinctnessCodeToOffset(sp1$bound_distinct)
plot(sp1, hz.distinctness.offset='hzD')

# plot horizon color according to some property
data(sp4)
depths(sp4) <- id ~ top + bottom
plot(sp4, color='clay')

# another example
data(sp2)
depths(sp2) <- id ~ top + bottom
site(sp2) <- ~ surface

# label with site-level attribute: `surface`
plot(sp2, label='surface', plot.order=order(sp2$surface))

# example using a categorical attribute
plot(sp2, color = "plasticity")

# plot two SPC objects in the same figure
par(mar=c(1,1,1,1))
# plot the first SPC object and
# allocate space for the second SPC object
plot(sp1, n=length(sp1) + length(sp2))
# plot the second SPC, starting from the first empty space
plot(sp2, x.idx.offset=length(sp1), add=TRUE)


##
## demonstrate adaptive legend
##

data(sp3)
depths(sp3) <- id ~ top + bottom

# make some fake categorical data
horizons(sp3)$fake.data <- sample(letters[1:15], size = nrow(sp3), replace=TRUE)

# better margins
par(mar=c(0,0,3,1))

# note that there are enough colors for 15 classes (vs. previous limit of 10)
# note that the legend is split into 2 rows when length(classes) > n.legend argument
plot(sp3, color='fake.data', name='fake.data', cex.names=0.8)

# make enough room in a single legend row
plot(sp3, color='fake.data', name='fake.data', cex.names=0.8, n.legend=15)
}
\references{
Beaudette, D.E., Roudier P., and A.T. O'Geen. 2013. Algorithms for Quantitative Pedology: A Toolkit for
Soil Scientists. Computers & Geosciences. 52:258 - 268.
}
\seealso{
\code{\link{fixOverlap}, \link{explainPlotSPC}, \link{SoilProfileCollection-class}, \link{pretty}, \link{hzDistinctnessCodeToOffset}, \link{addBracket}, \link{profileGroupLabels}}
}
\author{
D.E. Beaudette
}
\keyword{hplots}
