#' Using vector to visit vector binary tree
#'
#' @description Visit the vector binary tree and return a double list through specific assigment determined
#' by the argument \code{inq}.
#' @param x The vector binary tree to be visited. Traversal is available by setting -1 in desired layer.
#' @param inq An integer vector to determine desired location. The length of \code{inq} should be the same
#' as the layers of visited vector binary tree. If any assignment in specificed layer exceeds its intrinsic
#' length of visited vector binary tree layer, all elements will be returned in this layer.
#'
#' @return Return a double list according to the argument \code{inq}.
#' @export vbtinq
#' @seealso \code{\link[VBTree:vbtsub]{vbtsub}}, \code{\link[VBTree:advbtinq]{advbtinq}}, \code{\link[VBTree:advbtsub]{advbtsub}}.
#'
#' @examples
#' #Make vector binary tree:
#' colnamevbt <- dl2vbt(chrvec2dl(colnames(datatest)))
#'
#' #Visit by specific assignment:
#' vbtinq(colnamevbt, c(2, 3, 1, 1))
#'
#' #Traversal of the second layers:
#' vbtinq(colnamevbt, c(2, -1, 1, 1))
#'
#' #Invalid assignments in 1st and 3rd layers:
#' vbtinq(colnamevbt, c(4, 3, 7, 1))
#' @keywords Vector.Binary.Tree Double.List
vbtinq <- function(x, inq){
  # input diagnose
  if(class(x)!="Vector.Binary.Tree"){
    stop("x should be a Vector.Binary.Tree generated by arr2vbt(), ts2vbt() or dl2vbt().", call. = FALSE)
  }
  dims <- x$dims
  layers <- length(dims)

  # this inquiry method using vector: inq shuold be an integer numeric vector.
  if(is.vector(inq)!=TRUE | length(inq)!=layers){
    stop("inq must be a vector with the same length as the layers of x.", call. = FALSE)
  }

  inq <- as.integer(inq)

  # diagnose for available dimension assignment then make token in invalid layers
  if(any(inq > dims)){
    i <- which((inq - dims > 0)==TRUE)
    inq[i] <- -1
    warning(paste("The assignment in layer ", i, " is overflow. All elements in layer ", i, " will be returned completely.", "\n", sep = ""), call. = FALSE)
  }

  x_dl <- vbt2dl(x)

  # get all elements: if -1 appears in inq[i], export all elements in the i layers.
  prt <- inq != -1
  result <- x_dl

  i <- 1
  while (i <= layers) {
    if(prt[i]){
      result[[i]] <- x_dl[[i]][inq[i]]
    } else {
      result[[i]] <- x_dl[[i]]
    }
    i <- i + 1
  }
  return(result)
}

#' Using vector to generate sub tree from vector binary tree
#'
#' @description Visit the vector binary tree and generate a sub tree from visited vector binary tree, through
#' specific assigment determined by the argument \code{inq}.
#' @param x The vector binary tree to be visited. Traversal is available by setting -1 in desired layer.
#' @param inq An integer vector to determine the visiting location. The length of \code{inq} should be the same
#' as the layers of visited vector binary tree. If any assignment in specificed layer exceeds its intrinsic
#' length of visited vector binary tree layer, all elements will be returned in this layer.
#'
#' @return Return a sub tree from visited vector binary tree, according to the argument \code{inq}.
#' @export vbtsub
#' @seealso \code{\link[VBTree:vbtinq]{vbtinq}}, \code{\link[VBTree:advbtinq]{advbtinq}}, \code{\link[VBTree:advbtsub]{advbtsub}}.
#'
#' @examples
#' #Make vector binary tree:
#' colnamevbt <- dl2vbt(chrvec2dl(colnames(datatest)))
#'
#' #Generating sub tree by specific assignment:
#' vbtsub(colnamevbt, c(2, 3, 1, 1))
#'
#' #Generating sub tree with traversal in the second layers:
#' vbtsub(colnamevbt, c(2, -1, 1, 1))
#'
#' #Generating sub tree with invalid assignments in 1st and 3rd layers:
#' vbtsub(colnamevbt, c(4, 3, 7, 1))
#' @keywords Vector.Binary.Tree
vbtsub <- function(x, inq){
  result <- vbtinq(x, inq)
  result <- dl2vbt(result)
  return(result)
}
