\encoding{latin1}
\name{seqdef}
\alias{seqdef}
\alias{print.stslist}
\title{Create a state sequence object}
\description{
Create a state sequence object with attributes such as alphabet, color palette and state labels. Most TraMineR functions for state sequences require such a state sequence object as input argument. There are specific methods for plotting, summarizing and printing state sequence objects.
}

\usage{
seqdef(data, var, informat="STS", stsep="-",
	alphabet, states, start=1,
	left=NA, right="DEL", gaps=NA, missing=NA, void="\%", nr="*",
	cnames, cpal, missing.color="darkgrey", labels, ...)
}

\arguments{
  \item{data}{a data frame or matrix containing sequence data.}

  \item{var}{the list of columns containing the sequences. Defaut to NULL, ie all the columns. Whether the sequences are in the compressed (successive states in a character string) or extended format is automatically detected.}

   \item{informat}{format of the original data. Default is 'STS'. Avalaible formats are: STS, SPS, SPELL. See TraMineR user's manual (\cite{Gabadinho et al., 2008}) for a description of the formats.}

   \item{stsep}{the character used as separator in the original data if input format is successive states in a character string. By default, "-".}

  \item{alphabet}{optional vector containing the alphabet (the list of all possible states). Use this option if some states in the alphabet don't appear in the data or if you want to reorder the states. The specified vector MUST contain AT LEAST all the states appearing in the data. It may possibly contain additional states not appearing in the data. If NULL, the alphabet is set to the distinct states appearing in the data as returned by the \code{\link{seqstatl}} function.}

  \item{states}{an optional vector containing the labels for the states. Must have a length equal to the number of states in the data, and the labels must be ordered accordingly with the values returned by the \code{\link{seqstatl}} function.}

	\item{start}{starting time. For instance, if your sequences begin at age 15, you can specify 15. At this stage, used only for labelling column names.}
	\item{left}{the behavior for missing values appearing before the first (leftmost) valid state in each sequence. See \cite{Gabadinho et al. (2008)} for more details on the options for handling missing values when defining sequence objects. By default, left missing values are treated as 'real' missing values and converted to the internal missing value code defined by the \code{nr} option.}
	\item{right}{the behavior for missing values appearing after the last (rightmost) valid state in each sequence. See \cite{Gabadinho et al. (2008)} for more details on the options for handling missing values when defining sequence objects. By default, right missing values are treated as 'void' elements and converted to the internal code for void values defined by the \code{void} option.}
	\item{gaps}{the behavior for missing values appearing inside the sequences, i.e. after the first (leftmost) valid state and before the last (rightmost) valid state of each sequence. See \cite{Gabadinho et al. (2008)} for more details on the options for handling missing values when defining sequence objects. By default, gap missing values are treated as 'real' missing values and converted to the internal missing value code defined by the \code{nr} option.}
	\item{missing}{the code used for missing values in the input data. When specified, all cells containing this value will be replaced by NA's, the internal R code for missing values. If 'missing' is not specified, cells containing NA's are considered to be missing values.}
	\item{void}{the internal code used by TraMineR for representing void elements in the sequences. Default is "\code{\%}".}

	\item{nr}{the internal code used by TraMineR for representing real missing elements in the sequences. Default is "\code{*}".}

	\item{cnames}{optional names for the columns composing the sequence data. Those names will be used by default in the graphics as axis labels. When not specified, names are taken from the original column names in the data.}

	\item{cpal}{an optional color palette for representing the states in the graphics. If not specified, a color palette is created with the RColorBrewer package, using the "Accent" palette. Note that the maximum number of colors in this palette is 8. If the number of states in the data is greater than 8, you have to specify your own palette. The list of available colors is displayed by the \code{\link{colors}} function. You can also use alternatively some other palettes from the RColorBrewer package.}

	\item{missing.color}{alternative color for representing missing values inside the sequences. Defaults to "darkgrey".}

	\item{labels}{state labels used for the color legend of TraMineR's graphics.}

	\item{...}{options passed to the \code{\link{seqformat}} function for handling input data that is not in STS format.}
}

\details{Applying subscripts to sequence objects (eg. \code{seq[,1:5] or seq[1:10,]}) returns a state sequence object with some attributes preserved  (alphabet, missing) and some others (start, column names) adapted to the selected column or row subset. If only one column is specified, a factor is returned.}

\seealso{
 \code{\link{plot.stslist} to plot state sequence objects, \link{seqecreate}} to create an event sequence object.
}

\value{An object of class \code{stslist}. There are methods for \code{print}, \code{summary}, and subscripting sequence objects. State sequence objects are required as argument to other functions such as plotting functions (seqdplot, seqiplot or seqfplot), functions to compute distances (seqdist), etc...}

\references{Gabadinho, A., G. Ritschard, M. Studer and N. S. Mller (2008). Mining Sequence Data in \code{R} with \code{TraMineR}: A user's guide. \emph{Department of Econometrics and Laboratory of Demography, University of Geneva}.
}

\examples{
## Creating a sequence object with the columns 13 to 24
## in the 'actcal' example data set
data(actcal)
actcal.seq <- seqdef(actcal,13:24,
	labels=c("> 37 hours", "19-36 hours", "1-18 hours", "no work"))

## Displaying the first 10 rows of the sequence object
actcal.seq[1:10,]

## Displaying the first 10 rows of the sequence object
## in SPS format
print(actcal.seq[1:10,], format="SPS")

## Frequency plot for the monthes June to September
seqfplot(actcal.seq[,6:9], pbarw=TRUE)

## Re-ordering the alphabet
actcal.seq <- seqdef(actcal,13:24,alphabet=c("B","A","D","C"))
alphabet(actcal.seq)

## Adding a state not appearing in the data to the
## alphabet
actcal.seq <- seqdef(actcal,13:24,alphabet=c("A","B","C","D","E"))
alphabet(actcal.seq)

## Adding a state not appearing in the data to the
## alphabet and changing the states labels
actcal.seq <- seqdef(actcal,13:24,
  alphabet=c("A","B","C","D","E"),
  states=c("FT","PT","LT","NO","TR"))
alphabet(actcal.seq)
actcal.seq[1:10,]
}
\keyword{manip}
