% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/softbart_probit.R
\name{softbart_probit}
\alias{softbart_probit}
\title{SoftBart Probit Regression}
\usage{
softbart_probit(
  formula,
  data,
  test_data,
  num_tree = 20,
  k = 1,
  hypers = NULL,
  opts = NULL,
  verbose = TRUE
)
}
\arguments{
\item{formula}{A model formula with a binary factor on the left-hand-side and predictors on the right-hand-side.}

\item{data}{A data frame consisting of the training data.}

\item{test_data}{A data frame consisting of the testing data.}

\item{num_tree}{The number of trees in the ensemble to use.}

\item{k}{Determines the standard deviation of the leaf node parameters, which is given by \code{3 / k / sqrt(num_tree)}.}

\item{hypers}{A list of hyperparameters constructed from the \code{Hypers()} function (\code{num_tree}, \code{k}, and \code{sigma_mu} are overridden by this function).}

\item{opts}{A list of options for running the chain constructed from the \code{Opts()} function (\code{update_sigma} is overridden by this function).}

\item{verbose}{If \code{TRUE}, progress of the chain will be printed to the console.}
}
\value{
Returns a list with the following components:
\itemize{
  \item \code{sigma_mu}: samples of the standard deviation of the leaf node parameters
  \item \code{var_counts}: a matrix with a column for each predictor group containing the number of times each predictor is used in the ensemble at each iteration.
  \item \code{mu_train}: samples of the nonparametric function evaluated on the training set; \code{pnorm(mu_train)} gives the success probabilities.
  \item \code{mu_test}: samples of the nonparametric function evaluated on the test set; \code{pnorm(mu_train)} gives the success probabilities .
  \item \code{p_train}: samples of probabilities on training set.
  \item \code{p_test}: samples of probabilities on test set.
  \item \code{mu_train_mean}: posterior mean of \code{mu_train}.
  \item \code{mu_test_mean}: posterior mean of \code{mu_test}.
  \item \code{p_train_mean}: posterior mean of \code{p_train}.
  \item \code{p_test_mean}: posterior mean of \code{p_test}.
  \item \code{offset}: we fit model of the form (offset + BART), with the offset estimated empirically prior to running the chain.
  \item \code{pnorm_offset}: the \code{pnorm} of the offset, which is chosen to match the probability of the second factor level.
  \item \code{formula}: the formula specified by the user.
  \item \code{ecdfs}: empirical distribution functions, used by the \code{predict} function.
  \item \code{opts}: the options used when running the chain.
  \item \code{forest}: a forest object; see the \code{MakeForest} documentation for more details.
}
}
\description{
Fits a nonparametric probit regression model with the nonparametric function
modeled using a SoftBart model. Specifically, the model takes \eqn{\Pr(Y = 1
\mid X = x) = \Phi\{a + r(x)\}}{P(Y = 1 | X = x) = pnorm(a + r(x))} where
\eqn{a}{a} is an offset and \eqn{r(x)}{r(x)} is a Soft BART ensemble.
}
\examples{

## NOTE: SET NUMBER OF BURN IN AND SAMPLE ITERATIONS HIGHER IN PRACTICE

num_burn <- 10 ## Should be ~ 5000
num_save <- 10 ## Should be ~ 5000

set.seed(1234)
f_fried <- function(x) 10 * sin(pi * x[,1] * x[,2]) + 20 * (x[,3] - 0.5)^2 + 
  10 * x[,4] + 5 * x[,5]

gen_data <- function(n_train, n_test, P, sigma) {
  X <- matrix(runif(n_train * P), nrow = n_train)
  mu <- (f_fried(X) - 14) / 5
  X_test <- matrix(runif(n_test * P), nrow = n_test)
  mu_test <- (f_fried(X_test) - 14) / 5
  Y <- factor(rbinom(n_train, 1, pnorm(mu)), levels = c(0,1))
  Y_test <- factor(rbinom(n_test, 1, pnorm(mu_test)), levels = c(0,1))
  
  return(list(X = X, Y = Y, mu = mu, X_test = X_test, Y_test = Y_test, 
              mu_test = mu_test))
}

## Simiulate dataset
sim_data <- gen_data(250, 250, 100, 1)

df <- data.frame(X = sim_data$X, Y = sim_data$Y)
df_test <- data.frame(X = sim_data$X_test, Y = sim_data$Y_test)

## Fit the model

opts <- Opts(num_burn = num_burn, num_save = num_save)
fitted_probit <- softbart_probit(Y ~ ., df, df_test, opts = opts)

## Plot results

plot(fitted_probit$mu_test_mean, sim_data$mu_test)
abline(a = 0, b = 1)


}
