#   R package for Singular Spectrum Analysis
#   Copyright (c) 2009 Anton Korobeynikov <asl@math.spbu.ru>
#
#   This program is free software; you can redistribute it
#   and/or modify it under the terms of the GNU General Public
#   License as published by the Free Software Foundation;
#   either version 2 of the License, or (at your option)
#   any later version.
#
#   This program is distributed in the hope that it will be
#   useful, but WITHOUT ANY WARRANTY; without even the implied
#   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#   PURPOSE.  See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public
#   License along with this program; if not, write to the
#   Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
#   MA 02139, USA.

#   Routines for toeplitz SSA

Lcor <- function(F, L) {
  storage.mode(F) <- "double"
  storage.mode(L) <- "integer"
  .Call("Lcor", F, L)
}

new.tmat <- function(F, L = (N + 1) %/% 2, fft.plan = NULL) {
  N <- length(F)
  R <- Lcor(F, L)

  storage.mode(R) <- "double"
  t <- .Call("initialize_tmat", R, if (is.null(fft.plan)) fft.plan.1d(2*L - 1) else fft.plan)
}

tcols <- function(t) {
  .Call("toeplitz_cols", t)
}

trows <- function(t) {
  .Call("toeplitz_rows", t)
}

is.tmat <- function(t) {
  .Call("is_tmat", t)
}

tmatmul <- function(tmat, v, transposed = FALSE) {
  storage.mode(v) <- "double"
  storage.mode(transposed) <- "logical"
  .Call("tmatmul", tmat, v, transposed)
}

.init.toeplitz.ssa <- function(x, ...) {
  # Initialize FFT plan
  .get.or.create.fft.plan(x)

  x
}

.hankelize.one.toeplitz.ssa <- .hankelize.one.1d.ssa

.get.or.create.tmat <- function(x) {
  .get.or.create(x, "tmat", new.tmat(F = x$F, L = x$window))
}

decompose.toeplitz.ssa.nutrlan <- function(x,
                                           neig = min(50, L, K),
                                           ...) {
  N <- x$length; L <- x$window; K <- N - L + 1

  F <- .get(x, "F")
  h <- .get.or.create.hmat(x)

  olambda <- .get(x, "olambda", allow.null = TRUE)
  U <- .get(x, "U", allow.null = TRUE)

  T <- .get.or.create.tmat(x)

  S <- trlan.eigen(T, neig = neig, ...,
                   lambda = olambda, U = U)

  # Save results
  .set(x, "olambda", S$d)
  if (!is.null(S$u))
    .set(x, "U", S$u)

  num <- length(S$d)
  lambda <- numeric(num)
  V <- matrix(nrow = K, ncol = num)
  for (i in 1:num) {
    Z <- hmatmul(h, S$u[, i], transposed = TRUE)
    lambda[i] <- sqrt(sum(Z^2))
    V[, i] <- Z / lambda[i]
  }

  # Save results
  .set(x, "lambda", lambda)
  .set(x, "V", V)

  x
}

decompose.toeplitz.ssa.eigen <- function(x,
                                         neig = min(50, L, K),
                                         ...,
                                         force.continue = FALSE) {
  N <- x$length; L <- x$window; K <- N - L + 1

  # Check, whether continuation of decomposition is requested
  if (!force.continue && nlambda(x) > 0)
    stop("Continuation of decompostion is not supported for this method.")

  # Build hankel matrix
  F <- .get(x, "F")
  h <- .get.or.create.hmat(x)

  # Do decomposition
  C <- toeplitz(Lcor(F, L))
  S <- eigen(C, symmetric = TRUE)

  .set(x, "U", S$vectors[, 1:neig])

  lambda <- numeric(L)
  V <- matrix(nrow = K, ncol = L)
  for (i in 1:L) {
    Z <- hmatmul(h, S$vectors[,i], transposed = TRUE)
    lambda[i] <- sqrt(sum(Z^2))
    V[, i] <- Z / lambda[i]
  }

  # Save results
  .set(x, "lambda", lambda[1:neig])
  .set(x, "V", V[, 1:neig, drop = FALSE])

  x
}

decompose.toeplitz.ssa.svd <- function(x,
                                       neig = min(50, L, K),
                                       ...,
                                       force.continue = FALSE) {
  N <- x$length; L <- x$window; K <- N - L + 1

  # Check, whether continuation of decomposition is requested
  if (!force.continue && nlambda(x) > 0)
    stop("Continuation of decompostion is not supported for this method.")

  # Build hankel matrix
  F <- .get(x, "F")
  h <- .get.or.create.hmat(x)

  # Do decomposition
  C <- toeplitz(Lcor(F, L))
  S <- svd(C, nu = neig, nv = neig)

  .set(x, "U", S$u)

  lambda <- numeric(neig)
  V <- matrix(nrow = K, ncol = neig)
  for (i in 1:neig) {
    Z <- hmatmul(h, S$u[,i], transposed = TRUE)
    lambda[i] <- sqrt(sum(Z^2))
    V[, i] <- Z / lambda[i]
  }

  # Save results
  .set(x, "lambda", lambda)
  .set(x, "V", V)

  x
}

decompose.toeplitz.ssa.propack <- function(x,
                                           neig = min(50, L, K),
                                           ...,
                                           force.continue = FALSE) {
  N <- x$length; L <- x$window; K <- N - L + 1

  # Check, whether continuation of decomposition is requested
  if (!force.continue && nlambda(x) > 0)
    stop("Continuation of decompostion is not yet implemented for this method.");

  F <- .get(x, "F")
  h <- .get.or.create.hmat(x)

  olambda <- .get(x, "olambda", allow.null = TRUE)
  U <- .get(x, "U", allow.null = TRUE)

  T <- .get.or.create.tmat(x)

  S <- propack.svd(T, neig = neig, ...)

  # Save results
  .set(x, "olambda", S$d)
  if (!is.null(S$u))
    .set(x, "U", S$u)

  num <- length(S$d)
  lambda <- numeric(num)
  V <- matrix(nrow = K, ncol = num)
  for (i in 1:num) {
    Z <- hmatmul(h, S$u[, i], transposed = TRUE)
    lambda[i] <- sqrt(sum(Z^2))
    V[, i] <- Z / lambda[i]
  }

  # Save results
  .set(x, "lambda", lambda)
  .set(x, "V", V)

  x
}

decompose.toeplitz.ssa <- function(x,
                                   neig = min(50, L, K),
                                   ...,
                                   force.continue = FALSE) {
  N <- x$length; L <- x$window; K <- N - L + 1
  stop("Unsupported SVD method for Toeplitz SSA!")
}

calc.v.toeplitz.ssa <- function(x, idx, ...)
  x$V[, idx]
