% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/magic.R
\name{magic}
\alias{magic}
\alias{magic.default}
\alias{magic.seurat}
\alias{magic.Seurat}
\title{Perform MAGIC on a data matrix}
\usage{
magic(data, ...)

\method{magic}{default}(data, genes = NULL, knn = 5, knn.max = NULL,
  decay = 1, t = 3, npca = 100, init = NULL, t.max = 20,
  knn.dist.method = "euclidean", verbose = 1, n.jobs = 1,
  seed = NULL, k = NULL, alpha = NULL, ...)

\method{magic}{seurat}(data, genes = NULL, knn = 5, knn.max = NULL,
  decay = 1, t = 3, npca = 100, init = NULL, t.max = 20,
  knn.dist.method = "euclidean", verbose = 1, n.jobs = 1,
  seed = NULL, ...)

\method{magic}{Seurat}(data, assay = NULL, genes = NULL, knn = 5,
  knn.max = NULL, decay = 1, t = 3, npca = 100, init = NULL,
  t.max = 20, knn.dist.method = "euclidean", verbose = 1,
  n.jobs = 1, seed = NULL, ...)
}
\arguments{
\item{data}{input data matrix or Seurat object}

\item{...}{Arguments passed to and from other methods}

\item{genes}{character or integer vector, default: NULL
vector of column names or column indices for which to return smoothed data
If 'all_genes' or NULL, the entire smoothed matrix is returned}

\item{knn}{int, optional, default: 5
number of nearest neighbors on which to compute bandwidth}

\item{knn.max}{int, optional, default: NULL
maximum number of neighbors for each point. If NULL, defaults to 3*knn}

\item{decay}{int, optional, default: 1
sets decay rate of kernel tails.
If NULL, alpha decaying kernel is not used}

\item{t}{int, optional, default: 3
power to which the diffusion operator is powered
sets the level of diffusion. If 'auto', t is selected according to the
Procrustes disparity of the diffused data.'}

\item{npca}{number of PCA components that should be used; default: 100.}

\item{init}{magic object, optional
object to use for initialization. Avoids recomputing
intermediate steps if parameters are the same.}

\item{t.max}{int, optional, default: 20
Maximum value of t to test for automatic t selection.}

\item{knn.dist.method}{string, optional, default: 'euclidean'.
recommended values: 'euclidean', 'cosine'
Any metric from `scipy.spatial.distance` can be used
distance metric for building kNN graph.}

\item{verbose}{`int` or `boolean`, optional (default : 1)
If `TRUE` or `> 0`, print verbose updates.}

\item{n.jobs}{`int`, optional (default: 1)
The number of jobs to use for the computation.
If -1 all CPUs are used. If 1 is given, no parallel computing code is
used at all, which is useful for debugging.
For n_jobs below -1, (n.cpus + 1 + n.jobs) are used. Thus for
n_jobs = -2, all CPUs but one are used}

\item{seed}{int or `NULL`, random state (default: `NULL`)}

\item{k}{Deprecated. Use `knn`.}

\item{alpha}{Deprecated. Use `decay`.}

\item{assay}{Assay to use for imputation, defaults to the default assay}
}
\value{
If a Seurat object is passed, a Seurat object is returned. Otherwise, a "magic" object containing:
 * **result**: matrix containing smoothed expression values
 * **operator**: The MAGIC operator (python magic.MAGIC object)
 * **params**: Parameters passed to magic
}
\description{
Markov Affinity-based Graph Imputation of Cells (MAGIC) is an
algorithm for denoising and transcript recover of single cells
applied to single-cell RNA sequencing data, as described in
van Dijk et al, 2018.
}
\examples{
if (pymagic_is_available()) {

data(magic_testdata)

# Run MAGIC
data_magic <- magic(magic_testdata, genes=c("VIM", "CDH1", "ZEB1"))
summary(data_magic)
##       CDH1             VIM             ZEB1
## Min.   :0.4303   Min.   :3.854   Min.   :0.01111
## 1st Qu.:0.4444   1st Qu.:3.947   1st Qu.:0.01145
## Median :0.4462   Median :3.964   Median :0.01153
## Mean   :0.4461   Mean   :3.965   Mean   :0.01152
## 3rd Qu.:0.4478   3rd Qu.:3.982   3rd Qu.:0.01160
## Max.   :0.4585   Max.   :4.127   Max.   :0.01201

# Plot the result with ggplot2
if (require(ggplot2)) {
  ggplot(data_magic) +
    geom_point(aes(x=VIM, y=CDH1, color=ZEB1))
}

# Run MAGIC again returning all genes
# We use the last run as initialization
data_magic <- magic(magic_testdata, genes="all_genes", init=data_magic)
# Extract the smoothed data matrix to use in downstream analysis
data_smooth <- as.matrix(data_magic)

}

if (pymagic_is_available() && require(Seurat)) {

data(magic_testdata)

# Create a Seurat object
seurat_object <- CreateSeuratObject(counts = t(magic_testdata), assay="RNA")
seurat_object <- NormalizeData(object = seurat_object)
seurat_object <- ScaleData(object = seurat_object)

# Run MAGIC and reset the active assay
seurat_object <- magic(seurat_object)
seurat_object@active.assay = "MAGIC_RNA"

# Analyze with Seurat
VlnPlot(seurat_object, features=c("VIM", "ZEB1", "CDH1"))

}

}
