#' @title Mapping morphological convergence on 3D surfaces
#' @description Given vectors of RW (or PC) scores for some converging species,
#'   the function selects the RW (PC) axes which best account for convergence
#'   and maps convergent areas on the corresponding 3D surfaces.
#' @usage conv.map(x1,x2=NULL,scores, pcs, mshape,focal=NULL,mshape_sur=NULL,
#'   refmat = NULL,refsur = NULL, k = 4, exclude = NULL, out.rem = TRUE, plot =
#'   TRUE, col = "blue", NAcol = "gray", names = TRUE, nsim = 1000)
#' @param x1,x2 vectors of convergent species. When convergence within a single
#'   clade was found, \code{x1} represents the vector of species belonging to the clade
#'   (\code{x2=NULL}). When convergence between groups/clades was found,
#'   \code{x1} and \code{x2} are the two convergent groups/clades.
#' @param scores data frame (or matrix) with the RW (or PC) scores returned by
#'   RWA/PCA. Species not included in \code{x1} or \code{x2} are ignored.
#' @param pcs RW (or PC) vectors (eigenvectors of the covariance matrix)
#'   returned by RWA/PCA.
#' @param mshape the consensus configuration.
#' @param focal vector of species included in \code{x1/x2} to be plotted. To be
#'   provided if \code{refsur=NULL} and \code{refmat=NULL}.
#' @param mshape_sur a \code{mesh3d} object used as a reference for mesh
#'   reconstruction. The vertices of \code{mshape_sur} must be the consensus
#'   configuration. If \code{NULL}, it is automatically generated by applying
#'   \code{\link[Rvcg]{vcgBallPivoting}} on \code{mshape}.
#' @param refsur a named list of \code{mesh3d} objects for species in
#'   \code{x1/x2} to be plotted. If \code{focal} is not \code{NULL} this is
#'   ignored.
#' @param refmat a named list of landmark sets corresponding to \code{refsur}.
#' @param k the argument \code{k} passed to \code{\link{interpolMesh}}.
#' @param exclude integer: the index numbers of the RWs (or PCs) to be excluded
#'   from the comparison.
#' @param out.rem logical: if \code{TRUE} triangles with outlying area
#'   difference are removed.
#' @param plot logical: if \code{TRUE}, the pairwise comparisons are plotted.
#'   For more than 5 pairwise comparisons, the plot is not shown.
#' @param col character: the color for plotting.
#' @param NAcol the argument \code{NAcol} passed to \code{\link{col2mesh}}.
#' @param names logical: if \code{TRUE}, the names of the groups or species are
#'   displayed in the 3d plot.
#' @param nsim the number of iterations to evaluate significance.
#' @details After selecting the RW (PC) axes which best account for convergence,
#'   \code{conv.map} uses such axes (and related scores) within
#'   \code{\link[Morpho]{restoreShapes}} (\pkg{Morpho}) to reconstruct landmark
#'   matrices for each convergent species (in \code{x1/x2}). The reconstruction
#'   of species 3d surfaces is based on \code{mshape_sur}, either provided by
#'   the user or generated within the function. Finally, the area differences
#'   between corresponding triangles of reconstructed 3d meshes for each
#'   possible pair of convergent species are calculated. In the calculation of
#'   differences, the possibility to find and remove outliers is supplied
#'   (\code{out.rem=TRUE}, we suggest considering this possibility if the mesh
#'   may contain degenerate facets).
#'
#'   If the combination of \code{focal} species (or species within
#'   \code{refsur/refmat}) contains a number equal or lower then 5 items,
#'   \code{conv.map} returns a \code{rgl} plot mapping the convergence on the 3D
#'   models. If lists of \code{refsur/refmat} are not provided, the area
#'   differences are plotted onto reconstructed surfaces. If
#'   \code{refsur/refmat} are available, difference values are interpolated by
#'   means of \code{\link{interpolMesh}} to be plotted onto real surfaces. When
#'   species in either \code{x1} or \code{x2} are missing from \code{focal} or
#'   \code{refmat/refsur}, \code{conv.map} plots the reconstructed surface of
#'   the species having the smallest \code{$selected} angle with the focal (see
#'   \code{angle.compare} in the description of outputs).
#'
#'   \code{conv.map} further gives the opportunity to exclude some RW (or PC)
#'   axes from the analysis because, for example, in most cases the first axes
#'   are mainly related to high-order morphological differences driven by
#'   phylogeny and size variations.
#' @export
#' @seealso \href{../doc/search.conv.html}{\code{search.conv} vignette} ;
#'   \code{\link[Morpho]{relWarps}} ; \code{\link[Morpho]{procSym}}
#' @importFrom grDevices rainbow
#' @importFrom utils combn
#' @importFrom rgl layout3d
#' @return The function returns a list including:
#'   \itemize{\item\strong{$angle.compare}: a data frame including the real angles
#'   between species shape vectors \code{$real.angle}, the angles computed between vectors
#'   of the selected RWs (or PCs) \code{$selected}, the angles between vectors of the
#'   non-selected RWs (or PCs) \code{$others}, the differences \code{selected-others}
#'   and its p-values. \item\strong{$selected.pcs} RWs (or PCs) axes selected
#'   for convergence. \item\strong{$average.dist} symmetric matrix of pairwise
#'   distances between 3D surfaces. \item\strong{$suface1} list of colored
#'   surfaces representing convergence between mesh A and B charted on mesh A.
#'   \item\strong{$suface2} list of colored surfaces representing convergence
#'   between mesh A and B charted on mesh B. \item \strong{$scale} the value
#'   used to set the color gradient, computed as the maximum of all differences
#'   between each surface and the mean shape.}
#' @author Marina Melchionna, Antonio Profico, Silvia Castiglione, Carmela
#'   Serio, Gabriele Sansalone, Pasquale Raia
#' @references Schlager, S. (2017). \emph{Morpho and Rvcg–Shape Analysis in R:
#'   R-Packages for geometric morphometrics, shape analysis and surface
#'   manipulations.} In: Statistical shape and deformation analysis. Academic
#'   Press.
#' @references Melchionna, M., Profico, A., Castiglione, S., Serio, C., Mondanaro,
#'   A., Modafferi, M., Tamagnini, D., Maiorano, L. , Raia, P., Witmer, L.M.,
#'   Wroe, S., & Sansalone, G. (2021). A method for mapping morphological
#'   convergence on three-dimensional digital models: the case of the mammalian
#'   sabre-tooth. \emph{Palaeontology}, 64, 573–584. doi:10.1111/pala.12542
#' @examples
#'   \donttest{
#'   da<-"https://github.com/pasraia/RRmorph_example_data/raw/refs/heads/main/RRmorphdata.rda"
#'   download.file(url=da,destfile = paste0(tempdir(),"/RRmorphdata.rda"))
#'   load(paste0(tempdir(),"/RRmorphdata.rda"))
#'
#'   require(Morpho)
#'
#'   pca<-procSym(endo.set)
#'   ldm_homo<-endo.set[,,"Homo_sapiens"]
#'   sur_homo<-endo.sur[["Homo_sapiens"]]
#'   ldm_macaca<-endo.set[,,"Macaca_fuscata"]
#'   sur_macaca<-endo.sur[["Macaca_fuscata"]]
#'
#'
#'   # Convergence within group plotted on reconstructed surfaces
#'   cm1<-conv.map(x1=c("Pan_troglodytes","Gorilla_gorilla","Pongo_abelii"),
#'                 scores=pca$PCscores,pcs=pca$PCs,mshape=pca$mshape,
#'                 focal=c("Pan_troglodytes","Gorilla_gorilla"))
#'
#'   # Convergence between group plotted on reconstructed surfaces
#'   cm2<-conv.map(x1=c("Pongo_abelii"),x2=c("Alouatta_caraya"),
#'                 scores=pca$PCscores,pcs=pca$PCs,mshape=pca$mshape,
#'                 focal="Alouatta_caraya")
#'
#'   # Convergence within group plotted on real surfaces
#'   cm3<-conv.map(x1=c("Homo_sapiens","Gorilla_gorilla","Pongo_abelii"),
#'                 scores=pca$PCscores,pcs=pca$PCs,mshape=pca$mshape,
#'                 refsur=list("Homo_sapiens"=sur_homo),
#'                 refmat=list("Homo_sapiens"=ldm_homo))
#'
#'   # Convergence between group plotted on real surfaces
#'   cm3<-conv.map(x1=c("Homo_sapiens","Pongo_abelii"),x2=c("Macaca_fuscata"),
#'                 scores=pca$PCscores,pcs=pca$PCs,mshape=pca$mshape,
#'                 refsur=list("Homo_sapiens"=sur_homo,"Macaca_fuscata"=sur_macaca),
#'                 refmat=list("Homo_sapiens"=ldm_homo,"Macaca_fuscata"=ldm_macaca))
#'   }




conv.map<-function (x1,x2=NULL,
                    scores, pcs, mshape,
                    focal=NULL,mshape_sur=NULL,
                    refmat = NULL,refsur = NULL,
                    k = 4, exclude = NULL, out.rem = TRUE,
                    plot = TRUE, col = "blue", NAcol = "gray",
                    names = TRUE, nsim = 1000){
  # require(inflection)
  # require(ddpcr)
  # require(rgl)
  # require(Rvcg)
  # require(Morpho)

  misspacks<-sapply(c("inflection","ddpcr"),requireNamespace,quietly=TRUE)
  if(any(!misspacks)){
    stop("The following package/s are needed for this function to work, please install it/them:\n ",
         paste(names(misspacks)[which(!misspacks)],collapse=", "),
         call. = FALSE)
  }

  if(is.null(focal)&is.null(refmat)&is.null(refsur))
    stop("One of focal, refmat/refsur should be provided")
  if (any(!is.null(c(refsur, refmat))) & !all(!is.null(c(refsur,refmat))))
    stop("Please, provide both surface and its landmark configuration")
  if(!inherits(scores,"matrix")) scores<-as.matrix(scores)
  if (!is.null(refsur)) {
    if (!inherits(refsur, "list")|!inherits(refmat, "list")) stop("refsur and refmat must be lists")
    if (length(refsur) != length(refmat))
      stop("The objects refsur and refmat should have the same length")

    if(is.null(names(refsur))|is.null(names(refmat))) stop("refsur and/or refmat are missing names")
  }
  if (is.null(x2)) comb.df<-combn(sort(x1), 2) else comb.df<-t(expand.grid(x1,x2))
  if(!is.null(focal)) plotsp<-focal else plotsp<-names(refmat)


  colnames(pcs) <-colnames(scores) <- sapply(1:ncol(scores),function(x) paste("S",x, sep = ""))
  if(!is.null(exclude)) scores<-scores[,-exclude,drop=FALSE]

  selected <- list()
  for (kk in 1:ncol(comb.df)) {
    vec1 <- scores[match(comb.df[1, kk], rownames(scores)), ]
    vec2 <- scores[match(comb.df[2, kk], rownames(scores)), ]

    j.ang<-sapply(1:length(vec1),function(i)
      rad2deg(acos((vec1[-i] %*% vec2[-i])/(unitV(vec1[-i])*unitV(vec2[-i])))))
    names(j.ang) <- names(vec1)
    cutter <- inflection::ede(seq(1,length(j.ang)),
                              j.ang[order(j.ang,decreasing = TRUE)], 0)[1]
    if (cutter == 1) {
      cutter2 <- inflection::ede(seq(1,length(j.ang[-1])),
                                 j.ang[order(j.ang, decreasing = TRUE)][-1],
                                 0)[1]
      cutter <- cutter2 + 1
    }
    if (cutter > 0.5 * length(j.ang)) cutter = round(0.5 * (length(j.ang)), 0)
    main <- j.ang[order(j.ang, decreasing = TRUE)][seq(1:cutter)]
    selected[[kk]] <- names(main)
  }
  sele0 <- table(unlist(selected))
  if (length(which(sele0 > 1)) > 1)
    selected <- names(sele0[which(sele0 > 1)]) else
      selected <- names(sele0)
  # selected <- sapply(selected, function(x) paste("S",x,sep = ""))

  if (is.null(mshape_sur))
    mshape_sur <- vcgBallPivoting(mshape, radius = 0)

  surfs.sp<-list()
  for(j in 1:length(c(x1,x2))){
    vec <- scores[match(c(x1,x2)[j], rownames(scores)), ]
    temp <- restoreShapes(vec[which(names(vec)%in%selected)], pcs[,which(colnames(pcs)%in%selected)], mshape)
    sur <- mshape_sur
    sur$vb[1:3, ] <- t(temp)
    surfs.sp[[j]]<-sur
  }
  names(surfs.sp)<-c(x1,x2)

  surfs <- surfsd <-  res <-  res2 <- list()
  for (kk in 1:ncol(comb.df)) {
    vec1 <- scores[match(comb.df[1, kk], rownames(scores)), ]
    vec2 <- scores[match(comb.df[2, kk], rownames(scores)), ]

    ang <- rad2deg(acos((vec1 %*% vec2)/(unitV(vec1)*unitV(vec2))))
    a.sel <- rad2deg(acos((vec1[match(selected, names(vec1))] %*%
                             vec2[match(selected, names(vec2))])/
                            (unitV(vec1[match(selected, names(vec1))]) *
                               unitV(vec2[match(selected,names(vec2))]))))
    a.others <- rad2deg(acos((vec1[-match(selected, names(vec1))] %*%
                                vec2[-match(selected, names(vec2))])/
                               (unitV(vec1[-match(selected,names(vec1))]) *
                                  unitV(vec2[-match(selected,names(vec2))]))))
    ang.diff <- (a.sel - a.others)[1]
    nn <- length(selected)
    inn <- array()
    sell <- list()
    for (i in 1:nsim) {
      xx1 <- vec1[sample(seq(1,length(vec1)), nn)]
      xx2 <- vec2[which(names(vec2) %in% names(xx1))]
      xn1 <- vec1[which(!names(vec1) %in% names(xx1))]
      xn2 <- vec2[which(!names(vec2) %in% names(xx1))]
      a1 <- rad2deg(acos((xx1 %*% xx2)/(unitV(xx1) * unitV(xx2))))
      a2 <- rad2deg(acos((xn1 %*% xn2)/(unitV(xn1) * unitV(xn2))))
      sell[[i]] <- list(selected = names(xx1), angs = c(a1, a2), d = a1 - a2)
      # if (sum(sell[[i]]$selected %in% selected)==nn) inn[i] <- 0 else inn[i] <- 1
      ifelse(sum(sell[[i]]$selected %in% selected)==nn,0,1)->inn[i]
    }
    sell <- sell[which(inn>0)]
    p.sell <- 1 - sum(sapply(sell, "[[",3) > ang.diff)/length(sell)

    mapD.sur <- areadiff(surfs.sp[[match(comb.df[1, kk],names(surfs.sp))]],
                                   mshape_sur, out.rem = out.rem, fact = 1.5)
    mapD.sur1 <- areadiff(surfs.sp[[match(comb.df[2, kk],names(surfs.sp))]],
                                    mshape_sur, out.rem = out.rem, fact = 1.5)
    msurD <- mapD.sur[[3]] - mapD.sur1[[3]]
    msurD1 <- mapD.sur1[[3]] - mapD.sur[[3]]
    thr <- max(abs(c(msurD, msurD1)))
    res[[kk]] <- list(angle.compare = c(real.angle = ang[1],
                                        selected = a.sel, others = a.others, ang.diff = ang.diff,
                                        p.value = p.sell))
    res2[[kk]] <- list(average.dist = mean(abs(c(msurD, msurD1))),
                       thr = thr)
    surfs[[kk]] <- c(surfs.sp[match(comb.df[1, kk],names(surfs.sp))],
                     surfs.sp[match(comb.df[2, kk],names(surfs.sp))])
    surfsd[[kk]] <- list(msurD, msurD1)
    names(surfsd[[kk]])<-names(surfs[[kk]])
  }
  names(surfsd)<-names(surfs)<-names(res)<-names(res2)<-apply(comb.df,2,paste,collapse="-")


  thrs <- sapply(res2, "[[", 2)

  pale = colorRampPalette(c(col, "white", "white", "white",
                            "white"))
  v.pale = pale(1000)
  v.pale = v.pale[seq(1, 1000, by = 100) - 1]
  v.pale = c(rev(v.pale[-1]), pale(1), v.pale[-1])


  if (is.null(x2)) {
    psp.x1<-plotsp[which(plotsp%in%x1)]
    if(length(psp.x1)==0) {
      warning("species in focal or refmat/refsur are not in x1 or x2, plot is not returned",immediate. = TRUE)
      combplot<-NULL
    } else{
      if (length(plotsp)>length(psp.x1)) {
        warning("some species in focal are not in x1",immediate. = TRUE)
        plotsp<-plotsp[which(plotsp%in%psp.x1)]
      }
      if(length(psp.x1)==1){
        combplot<-sapply(psp.x1, function(jj)
          strsplit(names(which.min(sapply(res[grep(jj,names(res))], function(j)
            unname(j[[1]][2])))),"-")[[1]])
        plotsp<-unique(c(combplot))
      }else combplot<-combn(sort(psp.x1),2)
    }
  } else {
    psp.x1<-plotsp[which(plotsp%in%x1)]
    psp.x2<-plotsp[which(plotsp%in%x2)]

    if(length(plotsp)==0){
      warning("species in focal or refmat/refsur are not in x1 or x2, plot is not returned",immediate. = TRUE)
      combplot<-NULL
    } else {
      if (length(plotsp)>length(c(psp.x1,psp.x2))) {
        warning("some species in focal are not in x1 or x2",immediate. = TRUE)
        plotsp<-plotsp[which(plotsp%in%c(x1,x2))]
      }

      if(length(psp.x1)==0) {
        combplot<-sapply(psp.x2, function(jj)
          strsplit(names(which.min(sapply(res[grep(jj,names(res))], function(j)
            unname(j[[1]][2])))),"-")[[1]])
        plotsp<-unique(c(combplot))
      } else  if(length(psp.x2)==0) {
        combplot<-sapply(psp.x1, function(jj)
          strsplit(names(which.min(sapply(res[grep(jj,names(res))], function(j)
            unname(j[[1]][2])))),"-")[[1]])
        plotsp<-unique(c(combplot))
      } else combplot<-t(expand.grid(psp.x1,psp.x2))
    }
  }


  if(!is.null(combplot)){
    meshes1 <- meshes2 <- list()
    for (kk in 1:ncol(combplot)) {

      meshes <- list()
      for (i in 1:2) {
        if(!is.null(refsur)&&combplot[i,kk]%in%names(refsur)){

          rsur <- rotmesh.onto(refsur[[match(combplot[i,kk], names(refsur))]],
                               refmat[[match(combplot[i,kk], names(refmat))]],
                               vert2points(surfs[[match(paste(combplot[,kk],collapse = "-"),names(surfs))]][[i]]),
                               scale = TRUE)$mesh
          rmat <- rotmesh.onto(refsur[[match(combplot[i,kk], names(refsur))]],
                               refmat[[match(combplot[i,kk], names(refmat))]],
                               vert2points(surfs[[match(paste(combplot[,kk],collapse ="-"),names(surfs))]][[i]]),
                               scale = TRUE)$yrot

          values <- interpolMesh(surfs[[match(paste(combplot[,kk],collapse ="-"),names(surfs))]][[i]],
                                 values = surfsd[[match(paste(combplot[,kk],collapse ="-"),names(surfsd))]][[i]],
                                 rsur, rmat, k = k,element = "triangles")

          meshes[[i]] <- col2mesh(rsur, values, pal = v.pale,
                                  NAcol = NAcol)
        }    else {
          values <- tri2verts(surfs[[kk]][[i]], surfsd[[kk]][[i]])
          meshes[[i]] <- col2mesh(surfs[[kk]][[i]], values,
                                  pal = v.pale, NAcol = NAcol)
        }
      }
      meshes1[[kk]] <- meshes[[1]]
      meshes2[[kk]] <- meshes[[2]]
    }

    names(meshes1) <- apply(combplot, 2, function(x) paste(x[1],x[2], sep = "-"))
    names(meshes2) <- apply(combplot, 2, function(x) paste(x[2],x[1], sep = "-"))


    if (isTRUE(plot) & length(plotsp) <= 5) {
      matplot <- matrix(0, length(plotsp), length(plotsp))
      rownames(matplot)<-colnames(matplot)<-plotsp

      for (b in 1:ncol(combplot)) {
        matplot[which(rownames(matplot)==combplot[1,b]),which(colnames(matplot)==combplot[2,b])]<-b
        matplot[which(rownames(matplot)==combplot[2,b]),which(colnames(matplot)==combplot[1,b])]<-ncol(combplot)+b
      }

      diag(matplot)<-max(matplot)+seq(1,length(plotsp))

      open3d()
      layout3d(matplot, sharedMouse = TRUE)
      for (i in 1:length(meshes1)) {
        shade3d(meshes1[[i]], specular = "black")
        next3d()
      }
      for (i in 1:length(meshes2)) {
        shade3d(meshes2[[i]], specular = "black")
        next3d()
      }
      for (i in 1:length(plotsp)) {
        spheres3d(mshape, radius = 1e-19)
        mtext3d(text = paste(plotsp[i]), font = 2,
                     edge = "y", line = 3)

        if(i < length(plotsp)) next3d()

      }
      colmap_tot <- colorRampPalette(v.pale)
      plot(seq(-max(thrs), max(thrs), length.out = 200), rep(0,
                                                             200), main = "area differences legend", xlab = "",
           ylab = "", col = "white")
      abline(v = seq(-max(thrs), max(thrs), length.out = 200),
             col = colmap_tot(200), lwd = 5)
    }
  }else meshes1<-meshes2<-NULL




  angle.compare <- do.call(rbind, lapply(res, "[[", 1))
  angle.compare <- as.data.frame(angle.compare[order(angle.compare[,2]), ])
  selected.pcs <- selected
  average.dist <- sapply(res2, "[[", 1)
  mat <- matrix(NA, length(c(x1,x2)), length(c(x1,x2)))
  rownames(mat)<-colnames(mat)<-c(x1,x2)

  for (b in 1:ncol(comb.df)) {
    mat[which(rownames(mat)==comb.df[1,b]),which(colnames(mat)==comb.df[2,b])]<-average.dist[b]
    mat[which(rownames(mat)==comb.df[2,b]),which(colnames(mat)==comb.df[1,b])]<-average.dist[b]
  }

  diag(mat)<-0

  return(list(angle.compare = angle.compare, selected.pcs = selected.pcs,
              average.dist = mat, surfaces1 = meshes1, surfaces2 = meshes2,
              scale = max(thrs)))
}
