% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CalculateNonLinearParameters.R
\name{CalculateTimeLag}
\alias{CalculateTimeLag}
\title{Estimate an appropiate time lag for the Takens' vectors}
\usage{
CalculateTimeLag(HRVData, technique = c("acf", "ami"),
  method = c("first.e.decay", "first.zero", "first.minimum", "first.value"),
  value = 1/exp(1), lagMax = NULL, doPlot = TRUE, ...)
}
\arguments{
\item{HRVData}{Data structure that stores the beats register and information related to it.}

\item{technique}{The technique that we shall use to estimate the time lag. 
Allowed values are \emph{"acf"} and \emph{"ami"}.}

\item{method}{The method that we shall use to select the time lag (see the Details section). Available methods
are \emph{"first.zero"}, \emph{"first.e.decay"}, \emph{"first.minimum"} and \emph{"first.value"}.}

\item{value}{Numeric value indicating the value that the autocorrelation/AMI function must cross in order to
select the time lag. It is used only with the "first.value" method.}

\item{lagMax}{Maximum lag at which to calculate the acf/AMI.}

\item{doPlot}{Logical value. If TRUE (default value), a plot of the autocorrelation/AMI function is shown.}

\item{...}{Additional parameters for the \emph{acf} or the 
\emph{mutualInformation} functions (see \code{\link[nonlinearTseries]{mutualInformation}}).}
}
\value{
The estimated time lag.
}
\description{
Given a time series (timeSeries), an embedding dimension (m) and a 
time lag (timeLag), the \eqn{n^{th}} 
Takens' vector is defined as 
\deqn{T[n]={timeSeries[n], timeSeries[n+ timeLag],...timeSeries[n+m*timeLag]}.}
This function estimates an appropiate time lag by using the autocorrelation or the
average mutual information (AMI) function.
}
\details{
A basic criteria for estimating a proper time lag is based on the following reasoning:
if the time lag used to build the Takens' vectors is too small, the coordinates will
be too highly temporally correlated and the embedding will tend to cluster around 
the diagonal in the phase space. If the time lag is chosen too large, the resulting 
coordinates may be almost uncorrelated and the resulting embedding will be very complicated. 
Thus, the autocorrelation function can be used for  estimating an appropiate time lag of
a time series. However, it must be noted that the autocorrelation is a linear statistic,
and thus it does not take into account nonlinear dynamical correlations. To take into
account nonlinear correlations the average mutual information (AMI) can be used. 
Independently of the technique used to compute the correlation, the time lag can
 be selected in a variety of ways: 
\itemize{
   \item Select the time lag where the autocorrelation/AMI function decays to 0 
   (\emph{first.zero} method). This
   method is not appropriate for the AMI function, since it only takes positive values.
   \item Select the time lag where the autocorrelation/AMI function decays to 
   1/e of its value at zero (\emph{first.e.decay} method).
   \item Select the time lag where the autocorrelation/AMI function reaches 
   its first minimum (\emph{first.minimum} method).
   \item Select the time lag where the autocorrelation/AMI function decays to
    the value specified by the user (\emph{first.value} method and 
    \emph{value} parameter).
}
}
\note{
If the autocorrelation/AMI function does not cross the specifiged value, an error is thrown. This may be solved
by increasing the lag.max or selecting a higher value to which the autocorrelation/AMI function may decay.

This function is based on the \code{\link[nonlinearTseries]{timeLag}} function from the 
nonlinearTseries package.
}
\examples{
\dontrun{
data(HRVProcessedData)
HRVData = HRVProcessedData
HRVData = SetVerbose(HRVData,T)
timeLag = CalculateTimeLag(HRVData,technique = "ami")
embeddingDim = CalculateEmbeddingDim(HRVData,
                                     timeLag = timeLag,
                                     maxEmbeddingDim = 15)
}
}
\references{
H. Kantz  and T. Schreiber: Nonlinear Time series Analysis (Cambridge university press)
}
\seealso{
\code{\link[nonlinearTseries]{timeLag}},\code{\link[nonlinearTseries]{mutualInformation}} .
}

