#' Plot Simulated Tracks
#'
#' \code{plot_sim()} creates a plot that visualizes both simulated and actual movement trajectories. This function is useful for comparing the simulated tracks generated by \code{simulate_track()} with the observed trajectories to evaluate how well the simulation models represent real movement patterns.
#'
#' @param data A \code{track} R object, which is a list consisting of two elements:
#'    * \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
#'    * \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
#' @param sim A \code{track simulation} R object, where each object is a list of simulated trajectories stored as \code{track} R objects.
#' @param colours_sim A vector of colors for plotting each set of simulated trajectories. If \code{NULL}, the default color will be black (\code{"#000000"}).
#' @param alpha_sim A numeric value between 0 and 1 for the transparency level of simulated trajectories. The default is \code{0.1}.
#' @param lwd_sim A numeric value for the line width of the simulated trajectory lines. The default is \code{0.5}.
#' @param colours_act A vector of colors for plotting actual trajectories. If \code{NULL}, the default color will be black (\code{"#000000"}).
#' @param alpha_act A numeric value between 0 and 1 for the transparency level of actual trajectories. The default is \code{0.6}.
#' @param lwd_act A numeric value for the line width of the actual trajectory lines. The default is \code{0.8}.
#'
#' @details
#' The function uses \pkg{ggplot2} to create a plot with the following components:
#' - Simulated trajectories are displayed with paths colored according to the \code{colours_sim} parameter, with the specified transparency \code{alpha_sim} and line width \code{lwd_sim}.
#' - Actual trajectories are overlaid in the colors specified by \code{colours_act}, with a transparency level \code{alpha_act} and line width \code{lwd_act} to provide a clear comparison.
#'
#' @return A \code{ggplot} object displaying the simulated and actual trajectories.
#'
#' @section Logo:
#' \if{html}{\figure{Logo.png}{options: width=30\%}}
#'
#' @author Humberto G. Ferrón
#' @author humberto.ferron@uv.es
#' @author Macroevolution and Functional Morphology Research Group (www.macrofun.es)
#' @author Cavanilles Institute of Biodiversity and Evolutionary Biology
#' @author Calle Catedrático José Beltrán Martínez, nº 2
#' @author 46980 Paterna - Valencia - Spain
#' @author Phone: +34 (9635) 44477
#'
#' @examples
#' # Example 1: Simulate tracks using data from the Paluxy River
#' # Default model (Unconstrained movement)
#' simulated_tracks <- simulate_track(PaluxyRiver, nsim = 3)
#'
#' # Plot simulated tracks with default settings and actual tracks
#' plot_sim(PaluxyRiver, simulated_tracks)
#'
#' # Example 2: Simulate tracks using the "Directed" model, representing movement toward a
#' # resource
#' simulated_tracks_directed <- simulate_track(PaluxyRiver, nsim = 3, model = "Directed")
#'
#' # Plot simulated tracks with specific colors and transparency for "Directed" model
#' plot_sim(PaluxyRiver, simulated_tracks_directed,
#'   colours_sim = c("#E69F00", "#56B4E9"),
#'   alpha_sim = 0.4, lwd_sim = 1, colours_act = c("black", "black"), alpha_act = 0.7,
#'   lwd_act = 2
#' )
#'
#' # Example 3: Simulate tracks using the "Constrained" model, representing movement along
#' # a feature
#' simulated_tracks_constrained <- simulate_track(PaluxyRiver, nsim = 3, model = "Constrained")
#'
#' # Plot simulated tracks with a different color scheme and width for "Constrained" model
#' plot_sim(PaluxyRiver, simulated_tracks_constrained,
#'   colours_sim = c("#E69F00", "#56B4E9"),
#'   alpha_sim = 0.6, lwd_sim = 0.1, alpha_act = 0.5, lwd_act = 2
#' )
#'
#' # Example 4: Simulate tracks using the "Unconstrained" model (random exploratory
#' # movement)
#' simulated_tracks_unconstrained <- simulate_track(PaluxyRiver, nsim = 3, model = "Unconstrained")
#'
#' # Plot simulated tracks with default colors and increased transparency for "Unconstrained"
#' # model
#' plot_sim(PaluxyRiver, simulated_tracks_unconstrained,
#'   colours_sim = c("#E69F00", "#56B4E9"),
#'   alpha_sim = 0.2, lwd_sim = 1, colours_act = c("#E69F00", "#56B4E9"), alpha_act = 0.9,
#'   lwd_act = 2
#' )
#'
#' # Subsetting trajectories with four or more steps in the Mount Tom dataset
#' sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
#'
#' # Example 5: Simulate tracks using data from Mount Tom
#' simulated_tracks_mt <- simulate_track(sbMountTom, nsim = 3)
#'
#' # Plot simulated tracks with default settings and actual tracks from Mount Tom
#' plot_sim(sbMountTom, simulated_tracks_mt)
#'
#' # Example 6: Simulate tracks using the "Directed" model for Mount Tom
#' simulated_tracks_mt_directed <- simulate_track(sbMountTom, nsim = 3, model = "Directed")
#'
#' # Plot simulated tracks with specific colors and transparency for "Directed" model for Mount
#' # Tom
#' plot_sim(sbMountTom, simulated_tracks_mt_directed, colours_sim = c(
#'   "#E69F00", "#56B4E9",
#'   "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7", "#999999", "#F4A300",
#'   "#6C6C6C", "#1F77B4"
#' ), alpha_sim = 0.3, lwd_sim = 1.5, alpha_act = 0.8, lwd_act = 2)
#'
#' # Example 7: Simulate tracks using the "Constrained" model for Mount Tom
#' simulated_tracks_mt_constrained <- simulate_track(sbMountTom, nsim = 3, model = "Constrained")
#'
#' # Plot simulated tracks with different color scheme and increased line width for "Constrained"
#' # model
#' plot_sim(sbMountTom, simulated_tracks_mt_constrained, colours_sim = c(
#'   "#E41A1C", "#377EB8",
#'   "#4DAF4A", "#FF7F00", "#F781BF", "#A65628", "#FFFF33", "#8DD3C7", "#FB8072",
#'   "#80BF91", "#F7F7F7"
#' ), alpha_sim = 0.5, lwd_sim = 0.2, alpha_act = 0.6, lwd_act = 2)
#'
#' # Example 8: Simulate tracks using the "Unconstrained" model for Mount Tom
#' simulated_tracks_mt_unconstrained <- simulate_track(sbMountTom, nsim = 3, model = "Unconstrained")
#'
#' # Plot simulated tracks with a different color scheme and transparency for "Unconstrained" model
#' plot_sim(sbMountTom, simulated_tracks_mt_unconstrained, colours_sim = c(
#'   "#6BAED6", "#FF7F00",
#'   "#1F77B4", "#D62728", "#2CA02C", "#9467BD", "#8C564B", "#E377C2", "#7F7F7F",
#'   "#BCBD22", "#17BECF"
#' ), alpha_sim = 0.2, lwd_sim = 0.5, colours_act = c(
#'   "#6BAED6",
#'   "#FF7F00", "#1F77B4", "#D62728", "#2CA02C", "#9467BD", "#8C564B", "#E377C2",
#'   "#7F7F7F", "#BCBD22", "#17BECF"
#' ), alpha_act = 1, lwd_act = 2)
#'
#' @importFrom ggplot2 ggplot geom_path coord_fixed theme_classic theme scale_color_manual xlab ylab
#'
#' @seealso \code{\link{tps_to_track}}, \code{\link{simulate_track}}
#'
#' @export

plot_sim <- function(data, sim, colours_sim = NULL, alpha_sim = NULL, lwd_sim = NULL, colours_act = NULL, alpha_act = NULL, lwd_act = NULL) {

  ## Set default values if arguments are NULL----
  if (is.null(colours_sim)) colours_sim <- rep("#000000", length(sim[[1]])) # Default to black for all simulated trajectories if no colors are provided
  if (is.null(alpha_sim)) alpha_sim <- 0.1 # Default transparency level for simulated trajectories
  if (is.null(lwd_sim)) lwd_sim <- 0.5 # Default line width for simulated trajectories
  if (is.null(colours_act)) colours_act <- rep("#000000", length(data[[1]])) # Default to black for all actual trajectories if no colors are provided
  if (is.null(alpha_act)) alpha_act <- 0.6 # Default transparency level for actual trajectories
  if (is.null(lwd_act)) lwd_act <- 0.8 # Default line width for actual trajectories


  ## Errors and Warnings----

  # Check if 'data' is a list with at least two elements
  if (!is.list(data) || length(data) < 2) {
    stop("The 'data' argument must be a 'track' R object, which is a list consisting of two elements.")
  }

  # Check if the two elements of 'data' are lists
  if (!is.list(data[[1]]) || !is.list(data[[2]])) {
    stop("The two elements of 'data' must be lists.")
  }

  # Check if 'sim' is a list of lists
  if (!is.list(sim) || !all(sapply(sim, is.list))) {
    stop("The 'sim' argument must be a 'track simulation' R object, where each object is a list of simulated trajectories.")
  }

  # Check colors for simulated tracks
  if (is.null(colours_sim)) {
    warning("The 'colours_sim' parameter is NULL. Defaulting to black ('#000000') for all simulated trajectories.")
  } else if (length(colours_sim) != length(sim[[1]])) {
    stop("The length of 'colours_sim' does not match the number of unique simulated trajectories. Please ensure 'colours_sim' has as many colors as there are unique simulated trajectories.")
  }

  # Check colors for actual tracks
  if (is.null(colours_act)) {
    warning("The 'colours_act' parameter is NULL. Defaulting to black ('#000000') for all actual trajectories.")
  } else if (length(colours_act) != length(data[[1]])) {
    stop("The length of 'colours_act' does not match the number of unique actual trajectories. Please ensure 'colours_act' has as many colors as there are unique actual trajectories.")
  }

  # Check and correct alpha values for transparency
  if (alpha_sim < 0 || alpha_sim > 1) {
    stop("Alpha value for simulated trajectories 'alpha_sim' should be between 0 and 1.")
  }

  if (alpha_act < 0 || alpha_act > 1) {
    stop("Alpha value for actual trajectories 'alpha_act' should be between 0 and 1.")
  }

  # Check and correct line width values
  if (lwd_sim <= 0) {
    stop("Line width value for simulated trajectories 'lwd_sim' should be a positive number.")
  }

  if (lwd_act <= 0) {
    stop("Line width value for actual trajectories 'lwd_act' should be a positive number.")
  }


  ## Code----

  # Extract trajectory data
  DATA <- data[[1]]

  # Combine simulated trajectories into a single data frame
  listSIM <- list()
  for (i in 1:length(sim)) {
    listSIM[[i]] <- bind_rows(sim[[i]])
  }
  SIM <- listSIM

  # Combine actual trajectories into a single data frame
  matrixact <- bind_rows(DATA)
  matrixact$ID <- as.factor(matrixact$ID)

  # Combine simulated trajectories into a single data frame
  matrixsim <- bind_rows(SIM)

  # Create the plot
  plotsim <- ggplot() +
    geom_path(data = matrixsim, aes(x = x, y = y, group = TrajSim, color = as.factor(Trajectory)), alpha = alpha_sim, linewidth = lwd_sim) +
    geom_path(data = matrixact, aes(x = x, y = y, group = ID, color = as.factor(IMAGE)), linewidth = lwd_act, alpha = alpha_act) +
    coord_fixed() +
    theme_light() +
    theme(legend.position = "none") +
    xlab("m") +
    ylab("m") +
    scale_color_manual(values = c(colours_sim, colours_act))

  return(plotsim)
}
