% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IDW.R
\name{IDW}
\alias{IDW}
\title{Inverse distance weighted interpolation (IDW)}
\usage{
IDW(
  BD_Obs,
  BD_Coord,
  shapefile,
  grid_resolution,
  p = 2,
  n_round = NULL,
  training = 1,
  stat_validation = NULL,
  Rain_threshold = NULL,
  save_model = FALSE,
  name_save = NULL
)
}
\arguments{
\item{BD_Obs}{A \code{data.table} or \code{data.frame} containing observational data with the following structure:
\itemize{
\item The first column (\code{Date}): A \code{Date} object representing the observation date.
\item The remaining columns: Each column corresponds to a unique ground station, where the column name is the station identifier.
}

The dataset should be structured as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{> BD_Obs
# A data.table or data.frame with n rows (dates) and m+1 columns (stations + Date)
   Date        ST001  ST002  ST003  ST004  ...
   <date>      <dbl>  <dbl>  <dbl>  <dbl>  ...
1  2015-01-01    0      0      0      0    ...
2  2015-01-02    0      0      0     0.2   ...
3  2015-01-03   0.1     0      0     0.1   ...
}\if{html}{\out{</div>}}
\itemize{
\item Each station column contains numeric values representing observed measurements.
\item The column names (station identifiers) must be unique and match those in \code{BD_Coord$Cod} to ensure proper spatial referencing.
}}

\item{BD_Coord}{A \code{data.table} or \code{data.frame} containing the metadata of the ground stations. It must include the following columns:
#' - \code{"Cod"}:
Unique identifier for each ground station.
\itemize{
\item \code{"X"}:
Latitude of the station in UTM format.
\item \code{"Y"}:
Longitude of the station in UTM format.
}}

\item{shapefile}{A shapefile defining the study area, used to constrain the interpolation to the region of interest.
The shapefile must be of class \code{SpatVector} (from the \code{terra} package) and should have a UTM coordinate reference system.}

\item{grid_resolution}{A numeric value indicating the resolution of the interpolation grid in kilometers (\code{km}).}

\item{p}{'Numeric' value that controls how the influence decreases with distance. The default value is 2}

\item{n_round}{An integer specifying the number of decimal places to round the interpolated results.
If set to \code{NULL}, all decimal places will be preserved. The default value is \code{1}.}

\item{training}{Numerical value between 0 and 1 indicating the proportion of data used for model training. The remaining data are used for validation. Note that if you enter, for example, 0.8 it means that 80 \% of the data will be used for training and 20 \% for validation.
If you do not want to perform validation, set training = 1. (Default training = 1).}

\item{stat_validation}{A character vector specifying the names of the stations to be used for validation.
This option should only be filled in when it is desired to manually enter the stations used for validation. If this parameter is NULL, and the formation is different from 1, a validation will be performed using random stations.
The vector must contain the names of the stations selected by the user for validation.
For example, stat_validation = c(“ST001”, “ST002”). (Default stat_validation = NULL).}

\item{Rain_threshold}{List of numerical vectors defining precipitation thresholds to classify precipitation into different categories according to its intensity.
This parameter should be entered only when the validation is to include categorical metrics such as Critical Success Index (CSI), Probability of Detection (POD), False Alarm Rate (FAR), etc.
Each list item should represent a category, with the category name as the list item name and a numeric vector specifying the lower and upper bounds of that category.
\strong{Note:} See the "Notes" section for additional details on how to define categories, use this parameter for validation, and example configurations.}

\item{save_model}{Logical value indicating whether the interpolation file should be saved to disk. The default value is \code{FALSE}. indicating that the interpolated file should not be saved.}

\item{name_save}{Character string indicating the name under which the interpolation raster file will be saved. By default the algorithm sets as output name: 'Model_IDW'.}
}
\value{
The return value will depend on whether validation has been performed or not.
If validation is not performed, the function will return a \code{SpatRaster} object with the interpolated values.
If validation is performed, the function will return a list with two elements:
\itemize{
\item \code{Ensamble}: A \code{SpatRaster} object with the interpolated values.
\item \code{Validation}: A \code{data.table} with the validation results, including goodness-of-fit metrics and categorical metrics (if \code{Rain_threshold} is provided).
}
}
\description{
This function performs Inverse Distance Weighting (IDW), which is a spatial interpolation method that estimates unknown values using the known values of surrounding points, assigning greater influence to closer points.
}
\section{Details}{

Inverse distance weighting (IDW) works as a deterministic mathematical interpolator that assumes
that values closer to an unknown point are more closely related to it than those farther away. When using this method, sample points are weighted during the interpolation process so that the influence of a known point on the unknown point decreases as the distance between them increases.
The IDW method calculates the value at an unknown point by a weighted average of the values of the known points, where the weights are inversely proportional to the distances between the prediction point and the known points. The basic formula defined by Shepard states that:
\deqn{\hat{Z}(s_0) = \frac{\sum_{i=1}^{N} w_i Z(s_i)}{\sum_{i=1}^{N} w_i}}
where:
\describe{
\item{\eqn{\hat{Z}(s_0)}}{ is the estimated value at the unknown point.}
\item{\eqn{Z(s_i)}}{ are the known values.}
\item{\eqn{w_i}}{ are the weights assigned to each known point.}
\item{\eqn{N}}{ is the total number of known points used.}
}

The weights are calculated by:
\deqn{w_i = \frac{1}{d(s_0, s_i)^p}}
where:
\describe{
\item{\eqn{d(s_0, s_i)}}{ is the distance between the unknown point \eqn{s_0} and the known point \eqn{s_i}.}
\item{\eqn{p}}{ is the power parameter that controls how the influence decreases with distance.}
}
The \code{Rain_threshold} parameter is used to calculate categorical metrics such as the Critical Success Index (CSI),
Probability of Detection (POD), False Alarm Rate (FAR), success ratio (SR), Hit BIAS (HB),Heidke Skill Score (HSS);
Hanssen-Kuipers Discriminant (HK); Equal Threat Score (ETS) or Gilbert Skill Score.
The parameter should be entered as a named list, where each item represents a category and the name of the item is the category name.
The elements of each category must be a numeric vector with two values: the lower and upper limits of the category.
For example:
#' \code{Rain_threshold = list(
  no_rain = c(0, 1),
  light_rain = c(1, 5),
  moderate_rain = c(5, 20),
  heavy_rain = c(20, 40),
  violent_rain = c(40, Inf)
)}

Precipitation values will be classified into these categories based on their intensity.
Users can define as many categories as necessary, or just two (e.g., "rain" vs. "no rain").
It is important that these categories are entered according to the study region, as each study region may have its own categories.
}

\examples{
\donttest{
library(InterpolateR)
# Load data from on-site observations
 data("BD_Obs", package = "InterpolateR")
 data("BD_Coord", package = "InterpolateR")

# Load the study area where the interpolation is performed.
 shapefile <- terra::vect(system.file("extdata/study_area.shp", package = "InterpolateR"))

 # Perform the interpolation
 Interpolated_data <- IDW(BD_Obs, BD_Coord, shapefile, grid_resolution = 5, p = 2,
                          n_round = 1, training = 0.8, Rain_threshold = NULL,
                          stat_validation = NULL, save_model = FALSE, name_save = NULL)
}
}
\references{
Shepard, D. (1968) A two-dimensional interpolation function for irregularly-spaced data. Proceedings of the 1968 ACM National Conference, 1968, pages 517--524. DOI: 10.1145/800186.810616
}
\author{
Jonnathan Landi \href{mailto:jonnathan.landi@outlook.com}{jonnathan.landi@outlook.com}
}
