% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simuRNAseq.R
\name{simuRNAseq}
\alias{simuRNAseq}
\title{Simulating Multi-sample Multi-cell-type scRNA-seq Dataset based on Negative Binomial Distribution}
\usage{
simuRNAseq(
  counts,
  nGenes = nrow(counts),
  nCells = ncol(counts),
  metadata = NULL,
  samples.nested = TRUE,
  nsam = 25,
  ncls = 10,
  ntrt = 2,
  trt = NULL,
  nDEgenes = ncls,
  nDEgenesType,
  pDEgenesType = NULL,
  adjust.library.size = TRUE,
  direction = c("both", "up", "down"),
  minbeta = 0.25,
  maxbeta = 1,
  var.randomeffects = 0.1
)
}
\arguments{
\item{counts}{A genes-by-cells matrix of reference counts. If missing, counts is generated by a negative binomial distribution.}

\item{nGenes}{Number of genes to be simulated.}

\item{nCells}{Number of cells to be simulated.}

\item{metadata}{The meta data consisting of 4 columns: sam (sample labels), cls (cluster lables of cell types), trt (treatments or conditions) and libsize (library size or total counts per cell), which is randomly generated if not provided.}

\item{samples.nested}{If TRUE, when metadata is not provided, each simulated subject (sample) belongs to only one condition (either treatment or control), that is, the subject is nested in a condition (treatment).}

\item{nsam}{Number of subjects (individuals).}

\item{ncls}{Number of clusters (cell-types).}

\item{ntrt}{Number of treatments (only one condition is treated).}

\item{trt}{Treatment, specifying which condition is treated.}

\item{nDEgenes}{Total number of DE genes.}

\item{nDEgenesType}{Number of DE genes specific to a cell type, named by cell cluster labels.}

\item{pDEgenesType}{Proportion of DE genes in a cell-type. Default NULL means equal proportion.}

\item{adjust.library.size}{If TRUE, adjust library sizes using the reference counts.}

\item{direction}{Specify if the DE genes are up- and/or down-regulated.}

\item{minbeta}{Lower bound of the DE gene logFC.}

\item{maxbeta}{Upper bound of the DE gene logFC. minbeta < maxbeta. If direction = "both", minbeta*maxbeta > 0. If direction = "up", minbeta > 0. If direction = "down", maxbeta < 0.}

\item{var.randomeffects}{Variance of random effects}
}
\value{
ref.mean.dispersion: A data frame of the reference counts' means and dispersion,

metadata: Meta data consisting of 4 columns: sam (sample labels), cls (cluster lables of cell types), trt (two treatments/conditions) and libsize (library sizes).

counts: A genes-by-cells matrix of the simulated counts.

DEgenes: A data frame of DE genes consisting of 3 columns: gene, beta (effect), and cluster to which the gene is specific.

treatment: The condition treated.
}
\description{
simuRNAseq simulates scRNA-seq data with multiple subjects (samples), multiple clusters (cell-types) and two treatments (conditions) based on a negative binomial (NB) distribution using a reference data as background or control.
The reference data consisting of genes-by-cells counts matrix is used to estimate the NB dispersion and means for the genes to be simulated. The simulated genes are randomly selected from the reference data.
The NB dispersion are estimated by the method-of-moments estimate (MME).
The NB means for the background in the control are estimated by the sample mean. The NB means for the differentially expressed (DE) genes are given by the sample mean plus a log-fold change (logFC).
The simulated cells are randomly selected from the meta data that specifies subjects, cell-types and treatments for the cells.
The meta data consists of samples, clusters of cell types, and treatments, which can be generated either from reference data or randomly. If not provided, it will be randomly generated.
A random seed is recommended to be specified by set.seed before simulating data.
}
\examples{
#Simulate a multi-sample multi-cell-type scRNA-seq dataset.
set.seed(2412)
dat <- simuRNAseq(nGenes = 50, nCells = 1000, nsam = 25, ncls = 4, ntrt = 2, nDEgenes = 6)
str(dat)
table(dat$metadata[, c("sam", "trt")]) #The samples are nested in a condition.

#Analyze differentially expressed (DE) genes specific to a cell-type using LMM.
Y <- log(dat$counts + 1) #expressions (log-transformed counts)
X <- model.matrix(~ 0 + log(libsize) + cls + cls:trt, data = dat$metadata)
Z <- model.matrix(~ 0 + sam, data = dat$metadata)
d <- ncol(Z)

#Fit LMM using cell-level data.
fit <- lmmfit(Y, X, Z, d = d)

#Fit LMM using summary-level data.
#Compute and store the summary-level data:
n <- nrow(X)
XX <- t(X)\%*\%X
XY <- t(Y\%*\%X)
ZX <- t(Z)\%*\%X
ZY <- t(Y\%*\%Z)
ZZ <- t(Z)\%*\%Z
Ynorm <- rowSums(Y*Y)
fitss <- lmm(XX, XY, ZX, ZY, ZZ, Ynorm = Ynorm, n = n, d = d)
identical(fit, fitss)

#Hypothesis testing
test <- lmmtest(fit)
head(test)

#The DE genes specific to a cell-type.
tail(test[, grep(":", colnames(test))])
#The true DE genes
dat$DEgenes

}
