% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelling.R
\name{create_prediction_data}
\alias{create_prediction_data}
\title{Generate or Load Cached Predictors Data}
\usage{
create_prediction_data(
  country_code,
  country_shape,
  pop_raster,
  ur_raster,
  adm2_shape,
  cell_size = 5000,
  ignore_cache = FALSE,
  output_dir = here::here("03_outputs", "3a_model_outputs")
)
}
\arguments{
\item{country_code}{A string representing the country code (e.g., "KEN").}

\item{country_shape}{An `sf` object representing the country's administrative
boundaries.}

\item{pop_raster}{A `terra` raster object representing the population raster.}

\item{ur_raster}{A `terra` raster object representing the urban extent
raster.}

\item{adm2_shape}{An `sf` object representing the administrative level 2
boundaries.}

\item{cell_size}{An integer specifying the cell size for the prediction grid
in meters (default is 5000).}

\item{ignore_cache}{A boolean input which is set to determine whether
to ignore the existing cache and write over it. Default is set to FALSE.}

\item{output_dir}{A string specifying the directory where the predictors data
file should be saved (default is "03_outputs/3a_model_outputs").}
}
\value{
A data object (`predictor_data`) containing the generated predictors.
}
\description{
This function creates predictors data based on spatial inputs or loads cached
predictors data if the file already exists. It saves the generated data to a
specified directory for reuse and provides progress updates.
}
\examples{
\donttest{
tf <- file.path(tempdir(), "test_env")

# Initialize with normalized path
dir.create(tf, recursive = TRUE, showWarnings = FALSE)

init(
  r_script_name = "full_pipeline.R",
  cpp_script_name = "model.cpp",
  path = tf,
  open_r_script = FALSE
)

# Download shapefiles
download_shapefile(
  country_codes =  "COM",
  dest_file = file.path(
    tf, "01_data", "1c_shapefiles",
    "district_shape.gpkg"
  )
)

# Download population rasters from worldpop
download_pop_rasters(
  country_codes = "COM",
  dest_dir = file.path(tf, "01_data", "1b_rasters", "pop_raster")
)

# Extract urban extent raster
extract_afurextent(
  dest_dir = file.path(tf, "01_data", "1b_rasters", "urban_extent")
)


urban_raster <-  terra::rast(
  file.path(tf, "01_data", "1b_rasters",
            "urban_extent", "afurextent.asc"))

pop_raster <-  terra::rast(
  file.path(tf, "01_data", "1b_rasters", "pop_raster",
            "com_ppp_2020_constrained.tif")

)

adm2_sf <- sf::read_sf(
 file.path(tf, "01_data", "1c_shapefiles",
            "district_shape.gpkg"))

country_sf <- sf::st_union(adm2_sf)

predictors <- create_prediction_data(
  country_code =  "COM",
  country_shape = country_sf,
  pop_raster = pop_raster,
  ur_raster = urban_raster,
  adm2_shape = adm2_sf,
  cell_size = 5000,
  output_dir = file.path(
    tf, "03_outputs/3a_model_outputs"
  )
)
}

}
